import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'details', label: 'Details', icon: 'info' },
  { id: 'location', label: 'Location', icon: 'location_on' },
  { id: 'rules', label: 'Rules', icon: 'rule' }
];

const GeofenceView = ({ geofenceId, onClose }) => {
  const [geofenceData, setGeofenceData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('details');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    axios.get(`/geofences/${geofenceId}`)
      .then(res => {
        setGeofenceData(res.data);
        setLoading(false);
      })
      .catch(err => {
        console.error("Error fetching geofence:", err);
        setError('Error fetching geofence details.');
        setLoading(false);
      });
  }, [geofenceId]);

  const renderTabContent = () => {
    if (!geofenceData) return null;

    switch (activeTab) {
      case 'details':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                info
              </span>
              Geofence Details
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Name:</strong> {geofenceData.name}</p>
              <p><strong>Description:</strong> {geofenceData.description || 'N/A'}</p>
              <p><strong>Status:</strong> {geofenceData.status || 'Active'}</p>
              <p><strong>Created:</strong> {new Date(geofenceData.created_date).toLocaleDateString()}</p>
              <p><strong>Last Updated:</strong> {new Date(geofenceData.updated_date).toLocaleDateString()}</p>
            </div>
          </div>
        );

      case 'location':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                location_on
              </span>
              Location Details
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <p><strong>Shape:</strong> {geofenceData.shape}</p>
              
              {geofenceData.shape === 'circle' ? (
                <>
                  <p><strong>Center Coordinates:</strong> {geofenceData.center?.coordinates?.join(', ') || 'N/A'}</p>
                  <p><strong>Radius:</strong> {geofenceData.radius ? `${geofenceData.radius}m` : 'N/A'}</p>
                </>
              ) : (
                <>
                  <p><strong>Polygon Coordinates:</strong></p>
                  <div style={{ maxHeight: '200px', overflowY: 'auto', marginTop: '10px' }}>
                    {geofenceData.polygon_coordinates?.map((coord, index) => (
                      <div key={index} style={{ padding: '5px 0', borderBottom: '1px solid #f5f5f5' }}>
                        {coord.latitude}, {coord.longitude}
                      </div>
                    ))}
                  </div>
                </>
              )}
            </div>
          </div>
        );

      case 'rules':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333'
              }}>
                rule
              </span>
              Associated Rules
            </h4>

            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px',
              minHeight: '200px'
            }}>
              {geofenceData.rules?.length > 0 ? (
                <div>
                  {geofenceData.rules.map((rule, index) => (
                    <div key={index} style={{ 
                      padding: '10px',
                      borderBottom: index < geofenceData.rules.length - 1 ? '1px solid #eee' : 'none'
                    }}>
                      <p><strong>{rule.name}</strong></p>
                      <p>{rule.description || 'No description'}</p>
                    </div>
                  ))}
                </div>
              ) : (
                <div style={{ textAlign: 'center', color: '#666', padding: '20px' }}>
                  No rules associated with this geofence
                </div>
              )}
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading geofence data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
{/* Header */}
<div style={{
  display: 'flex',
  alignItems: 'center',
  gap: '20px',
  padding: '0px 10px 10px',
  borderBottom: '1px solid #e0e0e0'
}}>
  <div style={{
    width: '60px',
    height: '60px',
    borderRadius: '50%',
    backgroundColor: '#e0e0e0',
    color: '#000',
    display: 'flex',
    alignItems: 'center',
    justifyContent: 'center',
    fontSize: '24px',
    fontWeight: 'bold',
    textTransform: 'uppercase'
  }}>
    {geofenceData?.name?.charAt(0) || 'G'}
  </div>
  <div>
    <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '20px', color: 'black' }}>
      {geofenceData.name}
    </h3>
    <p style={{ marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
      ID: #{geofenceData.id}
    </p>
  </div>
</div>

        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '20px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-2px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Content */}
        <div style={{ padding: '0 20px', overflow: 'auto', maxHeight: '400px' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default GeofenceView;