

import React, { useState, useEffect } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import GeofenceView from './GeofenceView';
import GeofenceUpdate from './GeofenceUpdate';
import GeofenceDB from './GeofenceDB';

const GEOFENCE_HEADER_TO_KEY_MAP = {
  "Name": { key: "name", label: "Name", sortable: true },
  "Shape": { key: "shape", label: "Shape", sortable: true },
  "Radius": { key: "radius", label: "Radius (m)", sortable: false },
  // "Coordinates": { key: "coordinates", label: "Coordinates", sortable: false },
  "Created Date": { key: "created_date", label: "Created Date", sortable: true },
};

const GeofenceList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [accountId, setAccountId] = useState(null);
  const [userId, setUserId] = useState(null);
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery
  });
  const [selectedGeofence, setSelectedGeofence] = useState(null);
  
  useEffect(() => {
    const storedAccountId = localStorage.getItem('account_id');
    const storedUserId = localStorage.getItem('user_id');
    if (storedAccountId) setAccountId(storedAccountId);
    if (storedUserId) setUserId(storedUserId);
  }, []);

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'shape',
      type: 'dropdown',
      placeholder: 'Shape',
      options: [
        { label: 'All', value: '' },
        { label: 'Circle', value: 'circle' },
        { label: 'Polygon', value: 'polygon' }
      ]
    }
  ];

  const actionFormats = [
    { label: "view", modal: true, url: "/geofences/{id}" },
    { label: "edit", modal: true, url: "/geofences/{id}" },
    { label: "delete", modal: true, url: "/geofences/{id}" }
    
  ];

  const handleActionClick = (geofenceId, action) => {
    setSelectedGeofence({ geofenceId, modalType: action.label });
  };

  const buildFetchUrl = () => {
    if (!accountId) return '';
    let url = `/geofences/?account_id=${accountId}&skip=${searchParams.skip}&limit=${searchParams.limit}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.shape) url += `&shape=${searchParams.shape}`;
    return url;
  };

  if (!accountId || !userId) return <div>Loading...</div>;

  return (
    <div>
      <SearchPage
        fetchDataUrl={buildFetchUrl()}
        ListComponent={List}
        itemKeyMapping={GEOFENCE_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="id"
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        searchFields={searchFields}
      />

      {selectedGeofence?.modalType === "view" && (
        <GeofenceView geofenceId={selectedGeofence.geofenceId} onClose={() => setSelectedGeofence(null)} />
      )}
      {selectedGeofence?.modalType === "edit" && (
        <GeofenceUpdate geofenceId={selectedGeofence.geofenceId} onClose={() => setSelectedGeofence(null)} />
      )}
      {selectedGeofence?.modalType === "delete" && (
        <GeofenceDB geofenceId={selectedGeofence.geofenceId} onClose={() => setSelectedGeofence(null)} />
      )}
    </div>
  );
};

export default GeofenceList;