import React, { useState, useRef, useEffect } from 'react';
import { GoogleMap, LoadScript, Polygon, Circle, Marker } from '@react-google-maps/api';

const GeofenceForm = ({ onSubmit, initialData }) => {
  const [formData, setFormData] = useState({
    name: '',
    description: '',
    shape: 'circle',
    center: null, // Will be set dynamically
    radius: 1000,
    polygon_coordinates: [],
    ...initialData
  });
  const [errors, setErrors] = useState({});
  const [mapLoaded, setMapLoaded] = useState(false);
  const [isDrawing, setIsDrawing] = useState(false);
  const mapRef = useRef(null);

  // ✅ Dynamically set user location
  useEffect(() => {
    if (!formData.center) {
      navigator.geolocation.getCurrentPosition(
        (pos) => {
          const { latitude, longitude } = pos.coords;
          setFormData(prev => ({
            ...prev,
            center: { lat: latitude, lng: longitude }
          }));
        },
        (err) => {
          console.error('Geolocation error:', err.message);
          setFormData(prev => ({
            ...prev,
            center: { lat: 51.5074, lng: -0.1278 } // fallback to London
          }));
        }
      );
    }
  }, [formData.center]);

  const containerStyle = {
    width: '100%',
    height: '400px',
    borderRadius: '8px',
    overflow: 'hidden',
    position: 'relative'
  };

  const mapOptions = {
    disableDefaultUI: true,
    keyboardShortcuts: false,
    zoomControl: false,
    streetViewControl: false,
    gestureHandling: 'greedy',
    styles: [
      {
        featureType: 'poi',
        elementType: 'labels',
        stylers: [{ visibility: 'off' }]
      },
      {
        featureType: 'transit',
        elementType: 'labels',
        stylers: [{ visibility: 'off' }]
      }
    ]
  };

  const handleMapClick = (e) => {
    if (formData.shape === 'circle') {
      setFormData({
        ...formData,
        center: { lat: e.latLng.lat(), lng: e.latLng.lng() }
      });
    } else if (formData.shape === 'polygon' && isDrawing) {
      const newCoords = [...formData.polygon_coordinates,
        { lat: e.latLng.lat(), lng: e.latLng.lng() }
      ];
      setFormData({ ...formData, polygon_coordinates: newCoords });
    }
  };

  const toggleDrawing = () => setIsDrawing(!isDrawing);
  const clearPolygon = () => {
    setFormData({ ...formData, polygon_coordinates: [] });
    setIsDrawing(false);
  };
  
  const validateForm = () => {
    const newErrors = {};
    if (!formData.name.trim()) newErrors.name = 'Name is required';
    if (formData.shape === 'circle' && !formData.radius) newErrors.radius = 'Radius is required';
    if (formData.shape === 'polygon' && formData.polygon_coordinates.length < 3) newErrors.polygon = 'Minimum 3 points required';
    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    if (validateForm()) {
      const finalCoordinates = formData.shape === 'circle'
        ? [formData.center.lng, formData.center.lat]  // For circle: [lng, lat]
        : formData.polygon_coordinates.map(p => [p.lng, p.lat]);  // For polygon: array of [lng, lat] pairs

      onSubmit({
        ...formData,
        coordinates: finalCoordinates,
        // Remove the original coordinates data to avoid confusion
        center: undefined,
        polygon_coordinates: undefined
      });
    }
  };

  return (
    <div style={{ maxWidth: "600px", padding: "20px" }}>
      <form onSubmit={handleSubmit}>
        <div className="form-group">
          <label>Name *</label>
          <input
            type="text"
            value={formData.name}
            onChange={(e) => setFormData({ ...formData, name: e.target.value })}
            className={errors.name ? 'error' : ''}
          />
          {errors.name && <span className="error-message">{errors.name}</span>}
        </div>

        <div className="form-group">
          <label>Description</label>
          <textarea
            value={formData.description}
            onChange={(e) => setFormData({ ...formData, description: e.target.value })}
            rows="3"
          />
        </div>

        <div className="form-group shape-selector">
          <label>Shape *</label>
          <div className="radio-group">
            <label className={formData.shape === 'circle' ? 'active' : ''}>
              <input
                type="radio"
                value="circle"
                checked={formData.shape === 'circle'}
                onChange={() => {
                  setFormData({ ...formData, shape: 'circle' });
                  setIsDrawing(false);
                }}
              />
              Circle
            </label>
            <label className={formData.shape === 'polygon' ? 'active' : ''}>
              <input
                type="radio"
                value="polygon"
                checked={formData.shape === 'polygon'}
                onChange={() => setFormData({ ...formData, shape: 'polygon' })}
              />
              Polygon
            </label>
          </div>
        </div>

        {formData.shape === 'circle' ? (
          <div className="form-group">
            <label>Radius (meters) *</label>
            <input
              type="number"
              value={formData.radius}
              onChange={(e) => setFormData({ ...formData, radius: Number(e.target.value) })}
              className={errors.radius ? 'error' : ''}
              min="1"
            />
            {errors.radius && <span className="error-message">{errors.radius}</span>}
          </div>
        ) : (
          errors.polygon && <span className="error-message">{errors.polygon}</span>
        )}

        <div className="form-group map-container">
          <label>Draw on Map</label>
          <LoadScript
            googleMapsApiKey="AIzaSyDe-1PlmNj0J4kGAbZBJcelb_mlUof-6us"
            onLoad={() => setMapLoaded(true)}
          >
            {mapLoaded && (
              <div style={{ position: 'relative' }}>
                <GoogleMap
                  mapContainerStyle={containerStyle}
                  center={formData.center}
                  zoom={13}
                  onClick={handleMapClick}
                  options={mapOptions}
                  ref={mapRef}
                >
                  {formData.shape === 'circle' ? (
                    <>
                      <Circle
                        center={formData.center}
                        radius={formData.radius}
                        options={{
                          strokeColor: "#34A853", // Changed from #3367d6 to green (#34A853)
                          strokeOpacity: 0.8,
                          strokeWeight: 2,
                          fillColor: "#34A853", // Changed from #3367d6 to green (#34A853)
                          fillOpacity: 0.35,
                          draggable: true
                        }}
                        onDragEnd={(e) => {
                          setFormData({
                            ...formData,
                            center: { lat: e.latLng.lat(), lng: e.latLng.lng() }
                          });
                        }}
                      />
                      <Marker
                        position={formData.center}
                        icon={{
                          path: 'M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z',
                          fillColor: '#FF0000',
                          fillOpacity: 1,
                          strokeWeight: 0,
                          scale: 1.5,
                          anchor: new window.google.maps.Point(12, 24)
                        }}
                        draggable={true}
                        onDragEnd={(e) => {
                          setFormData({
                            ...formData,
                            center: { lat: e.latLng.lat(), lng: e.latLng.lng() }
                          });
                        }}
                      />
                    </>
                  ) : (
                    <>
                      <Polygon
                        paths={formData.polygon_coordinates}
                        options={{
                          strokeColor: "#34A853", // Changed from #4285F4 to green (#34A853)
                          strokeOpacity: 0.8,
                          strokeWeight: 2,
                          fillColor: "#34A853", // Changed from #4285F4 to green (#34A853)
                          fillOpacity: 0.2
                        }}
                      />
                      {formData.polygon_coordinates.map((position, i) => (
                        <Marker
                          key={i}
                          position={position}
                          icon={{
                            path: 'M19,13H13V19H11V13H5V11H11V5H13V11H19V13Z',
                            fillColor: '#34A853', // Changed from #4285F4 to green (#34A853)
                            fillOpacity: 1,
                            strokeWeight: 0,
                            scale: 1,
                            anchor: new window.google.maps.Point(12, 12)
                          }}
                        />
                      ))}
                    </>
                  )}
                </GoogleMap>

                {formData.shape === 'polygon' && (
                  <div className="map-controls">
                    <button
                      type="button"
                      onClick={toggleDrawing}
                      className={`icon-button ${isDrawing ? 'active' : ''}`}
                      title={isDrawing ? "Stop Drawing" : "Start Drawing"}
                    >
                      <span className="material-icons">
                        {isDrawing ? 'stop' : 'edit'}
                      </span>
                    </button>
                    {formData.polygon_coordinates.length > 0 && (
                      <button
                        type="button"
                        onClick={clearPolygon}
                        className="icon-button"
                        title="Clear Polygon"
                      >
                        <span className="material-icons">delete</span>
                      </button>
                    )}
                  </div>
                )}
              </div>
            )}
          </LoadScript>

          <div className="map-instructions">
            {formData.shape === 'circle' ? (
              <>
                <span className="material-icons">open_with</span>
                Click and drag to adjust circle position
              </>
            ) : (
              <>
                {isDrawing ? (
                  <>
                    <span className="material-icons">add_location</span>
                    Click map to add polygon points
                  </>
                ) : formData.polygon_coordinates.length > 0 ? (
                  <>
                    <span className="material-icons">edit</span>
                    Click edit icon to add more points
                  </>
                ) : (
                  <>
                    <span className="material-icons">touch_app</span>
                    Click edit icon to start drawing
                  </>
                )}
              </>
            )}
          </div>
        </div>

        {/* Updated Save Geofence button with green color and hover effects */}
        <div style={{ textAlign: 'left', marginTop: '1rem' }}>
  <button type="submit" className="submit-button">
    Save Geofence
  </button>
</div>
      </form>

      <style jsx>{`
        .geofence-form-container {
          max-width: 800px;
          margin: 2rem auto;
          padding: 2rem;
          background: #fff;
          border-radius: 12px;
          box-shadow: 0 1px 3px rgba(0,0,0,0.12);
        }

        h2 {
          color: #2d3748;
          margin-bottom: 1.5rem;
          display: flex;
          align-items: center;
          gap: 0.8rem;
          font-size: 1.5rem;
        }

        .form-group {
          margin-bottom: 1.5rem;
        }

        label {
          display: block;
          margin-bottom: 0.5rem;
          color: #4a5568;
          font-weight: 500;
        }

        input, textarea {
          width: 100%;
          padding: 0.75rem;
          border: 1px solid #e2e8f0;
          border-radius: 6px;
          font-size: 1rem;
          transition: border-color 0.2s;
        }

        input:focus, textarea:focus {
          outline: none;
          border-color: black; /* Changed from #4299e1 to green */
          box-shadow: 0 0 0 3px rgba(52, 168, 83, 0.1); /* Changed to green */
        }

        .radio-group {
          display: flex;
          gap: 1rem;
          margin-top: 0.5rem;
        }

        .radio-group label {
          display: flex;
          align-items: center;
          gap: 0.5rem;
          padding: 0.75rem 1rem;
          border: 1px solid #e2e8f0;
          border-radius: 6px;
          cursor: pointer;
          transition: all 0.2s;
        }

        .radio-group label.active {
          border-color: #34A853; /* Changed from #4299e1 to green */
          background-color: #e6f4ea; /* Changed from #ebf8ff to light green */
        }
          input[type="radio"] {
  accent-color: #34A853;
}

        .map-controls {
          position: absolute;
          top: 15px;
          left: 15px;
          display: flex;
          gap: 8px;
          z-index: 1;
        }

        .icon-button {
          display: flex;
          align-items: center;
          justify-content: center;
          width: 36px;
          height: 36px;
          border-radius: 50%;
          border: none;
          background: none;
          color: #5f6368;
          cursor: pointer;
          transition: color 0.2s;
          box-shadow: none;
        }

        .icon-button:hover {
          background: none !important;
          box-shadow: none !important;
          color: #34A853; /* Changed from #4285F4 to green */
        }

        .icon-button.active {
          background: none !important;
          box-shadow: none !important;
          color: #34A853; /* Changed from #4285F4 to green */
        }

        .map-instructions {
          margin-top: 1rem;
          display: flex;
          align-items: center;
          gap: 0.8rem;
          color: #718096;
          font-size: 0.9rem;
        }

        /* Updated submit button styles */
        .submit-button {
          display: flex;
          align-items: center;
          justify-content: center;
          gap: 0.8rem;
          width: 100%;
          padding: 1rem;
          background: white; /* Default background is white */
          color: #34A853; /* Default text color is green */
          border: 2px solid white; /* Green border */
          border-radius: 8px;
          font-size: 1rem;
          font-weight: bold;
          cursor: pointer;
          transition: all 0.2s;
        }
.submit-button {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.8rem;
    width: 150px; /* ✅ Fixed width instead of full width */
    padding: 10px 12px;
    background: white;
    color: #34A853;
    border: 1px solid white;
    border-radius: 4px;
    font-size: 1rem;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s ease;
  }

  .submit-button:hover {
    background: #34A853 !important;
    color: white !important;
  }

        .error {
          border-color: #fc8181;
        }

        .error-message {
          color: #fc8181;
          font-size: 0.875rem;
          margin-top: 0.25rem;
        }
      `}</style>

      <style jsx global>{`
        @import url('https://fonts.googleapis.com/icon?family=Material+Icons');
        .material-icons {
          font-family: 'Material Icons';
          font-weight: normal;
          font-style: normal;
          font-size: 24px;
          line-height: 1;
          letter-spacing: normal;
          text-transform: none;
          display: inline-block;
          white-space: nowrap;
          word-wrap: normal;
          direction: ltr;
          -webkit-font-smoothing: antialiased;
          text-rendering: optimizeLegibility;
          -moz-osx-font-smoothing: grayscale;
          font-feature-settings: 'liga';
        }
      `}</style>
    </div>
  );
};

export default GeofenceForm;