import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import VehicleModelView from './VehicleModelView';
import VehicleModelUpdate from './VehicleModelUpdate';
import VehicleModelDelete from './VehicleModelDelete';
import CommonData from '../../common/CommonData';

const VEHICLE_MODEL_HEADER_TO_KEY_MAP = {
  "Model Name": {
    key: "vehicle_type",
    label: "Model Name",
    sortable: true,
    clickable: true  // ✅ Make this clickable
  },
  "Manufacturer": { key: "manufacturer_name", label: "Manufacturer", sortable: true },
  "Fuel Type": { key: "fuel_type", label: "Fuel Type", sortable: true },
  "Autonomous": { key: "is_autonomous", label: "Autonomous", sortable: true },
  "Can Fly": { key: "can_fly", label: "Can Fly", sortable: true },
  "Job Capacity": { key: "capacity_jobs", label: "Job Capacity", sortable: true },
  "Wheels": { key: "number_of_wheels", label: "Wheels", sortable: false },
  "Max Speed": { key: "max_speed", label: "Max Speed", sortable: true },
  "Created Date": { key: "created_date", label: "Created Date", sortable: false },
  "Active Status": { key: "status", label: "Active Status", sortable: false },
};

const VehicleModelsList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const fetchDataUrl = `/vehicle-models/list/all/`;
  const [selectedModel, setSelectedModel] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  const enums = CommonData();

  const searchFields = [
    { name: 'created_date_from', type: 'date', placeholder: 'From Date' },
    { name: 'created_date_to', type: 'date', placeholder: 'To Date' },
      { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ],
    },
  ];

  const actionFormats = [
    { label: "view", modal: true, url: "/vehicle-models/view/{id}" },
    { label: "edit", modal: true, url: "/vehicle-models/update/{id}" },
    { label: "delete", modal: true, url: "/vehicle-models/delete/{id}" },
    
  ];

  const handleActionClick = (modelId, action) => {
    console.log(`Action clicked: ${action.label} for modelId: ${modelId}`);
    if (action.label === "view") {
      setSelectedModel({ modelId, modalType: "view" });
    } else if (action.label === "edit") {
      setSelectedModel({ modelId, modalType: "update" });
    } else if (action.label === "delete" || action.label === "block") {
      setSelectedModel({ modelId, modalType: "delete" });
    }
  };

  // ✅ Handles cell clicks (Model Name only)
  const handleCellClick = (header, item) => {
    console.log('🧩 handleCellClick → Header:', header);
    console.log('📦 Clicked Item:', item);

    if (header === "Model Name") {
      const modelId = item.vehicle_model_id || item.vehicle_id || item.id;
      if (modelId) {
        console.log('✅ Model Name clicked → Opening VehicleModelView for ID:', modelId);
        setSelectedModel({
          modelId: modelId,
          modalType: 'view',
          data: item
        });
      } else {
        console.warn('⚠️ No valid model ID found in item:', item);
      }
    }
  };

  const closeModal = () => setSelectedModel(null);

  const refreshList = () => {
    setRefreshTrigger(prev => prev + 1);
  };

  return (
    <div>
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={VEHICLE_MODEL_HEADER_TO_KEY_MAP}
        dataKey="vehicles"
        identifierKey="vehicle_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}  // ✅ Pass to enable clickable support
        refreshTrigger={refreshTrigger}
        searchFields={searchFields}
      />

      {selectedModel?.modalType === "view" && (
        <VehicleModelView
          modelId={selectedModel.modelId}
          modelData={selectedModel.data}  // optional: pass existing data
          onClose={closeModal}
        />
      )}

      {selectedModel?.modalType === "update" && (
        <VehicleModelUpdate
          modelId={selectedModel.modelId}
          onClose={closeModal}
          onUpdateSuccess={refreshList}
        />
      )}

      {selectedModel?.modalType === "delete" && (
        <VehicleModelDelete
          modelId={selectedModel.modelId}
          onClose={closeModal}
          onDeleteSuccess={refreshList}
        />
      )}
    </div>
  );
};

export default VehicleModelsList;
