import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'specs', label: 'Specifications', icon: 'description' },
  { id: 'features', label: 'Features', icon: 'star' },
  { id: 'usage', label: 'Usage Stats', icon: 'show_chart' },
  { id: 'fleet', label: 'Fleet Info', icon: 'directions_car' }
];

const VehicleModelView = ({ modelId, onClose }) => {
  const [modelData, setModelData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('specs');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    axios.get(`/vehicle-models/${modelId}`)
      .then(res => {
        setModelData(res.data);
        setLoading(false);
      })
      .catch(err => {
        setError('Error fetching model details');
        setLoading(false);
      });
  }, [modelId]);

  const renderTabContent = () => {
    if (!modelData) return null;

    switch (activeTab) {
      case 'specs':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>description</span>
              Technical Specifications
            </h4>
            <div className="specs-grid">
              <div className="detail-row"><strong>Manufacturer:</strong> {modelData.manufacturer_name}</div>
              <div className="detail-row"><strong>Type:</strong> {modelData.vehicle_type}</div>
              <div className="detail-row"><strong>Fuel Type:</strong> {modelData.fuel_type}</div>
              <div className="detail-row"><strong>Wheels:</strong> {modelData.number_of_wheels}</div>
              {modelData.max_speed && <div className="detail-row"><strong>Max Speed:</strong> {modelData.max_speed} km/h</div>}
              {modelData.engine_capacity && <div className="detail-row"><strong>Engine Capacity:</strong> {modelData.engine_capacity} cc</div>}
              <div className="detail-row"><strong>Job Capacity:</strong> {modelData.capacity_jobs}</div>
            </div>
          </div>
        );

      case 'features':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>star</span>
              Special Features
            </h4>
            <div className="features-grid">
              <div className="feature-card">
                <h5>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>auto_mode</span>
                  Autonomous
                </h5>
                <p>{modelData.is_autonomous ? 'Yes' : 'No'}</p>
              </div>
              <div className="feature-card">
                <h5>
                  <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>flight</span>
                  Flight Capable
                </h5>
                <p>{modelData.can_fly ? 'Yes' : 'No'}</p>
              </div>
            </div>
          </div>
        );

      case 'usage':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>show_chart</span>
              Usage Statistics
            </h4>
            <div className="stats-placeholder">
              <p>Charts and graphs showing model usage over time</p>
              <div className="stat-card">
                <h5>Active Vehicles</h5>
                <p className="stat-value">24</p>
              </div>
              <div className="stat-card">
                <h5>Average Utilization</h5>
                <p className="stat-value">78%</p>
              </div>
            </div>
          </div>
        );

      case 'fleet':
        return (
          <div className="tab-content">
            <h4>
              <span className="material-symbols-outlined" style={{ verticalAlign: 'middle', marginRight: '8px', color: '#000' }}>directions_car</span>
              Fleet Information
            </h4>
            <div className="fleet-info">
              <p><strong>Total in Fleet:</strong> 42</p>
              <p><strong>First Added:</strong> Jan 2021</p>
              <p><strong>Most Recent:</strong> Mar 2023</p>
            </div>
            <div className="fleet-locations">
              <h5>Common Locations</h5>
              <ul>
                <li>London Depot (15 vehicles)</li>
                <li>Manchester Yard (10 vehicles)</li>
                <li>Birmingham Hub (8 vehicles)</li>
              </ul>
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading vehicle model data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  return (
    <Modal onClose={onClose}>
      <div className="fixed top-0 right-0 w-full md:w-[55%] lg:w-[45%] xl:w-[40%] h-full bg-white shadow-lg z-50 overflow-hidden">
        {/* Header Section */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          padding: '10px 10px 20px',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px'
        }}>
          <div style={{ display: 'flex', alignItems: 'center' }}>
            <div style={{
              width: '60px',
              height: '60px',
              borderRadius: '50%',
              backgroundColor: '#e0e0e0',
              color: '#000',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              fontSize: '20px',
              fontWeight: 'bold',
              marginRight: '16px'
            }}>
              {modelData.manufacturer_name?.charAt(0).toUpperCase()}
            </div>
            <div>
              <div style={{
                fontWeight: '600',
                fontSize: '20px',
                lineHeight: '1.4',
                color: 'black'
              }}>
                {modelData.manufacturer_name} {modelData.model_name}
              </div>
              <div style={{
                fontSize: '16px',
                color: 'black',
                marginTop: '10px',
                fontWeight: 'bold'
              }}>
                Type: {modelData.vehicle_type}
              </div>
            </div>
          </div>
        </div>

        {/* Tabs Section */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '0px 10px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '20px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px',
                transition: 'min-width 0.2s ease'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && (
                <span>{tab.label}</span>
              )}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-1px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Tab Content */}
        <div style={{ maxHeight: 'calc(100vh - 250px)', overflowY: 'auto', padding: '0 20px' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default VehicleModelView;