import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import FleetView from './FleetView';
import FleetUpdate from './FleetUpdate';
import FleetDelete from './FleetDelete';
import CommonData from '../../common/CommonData';
import VendorView from '../Vendors/VendorView';
import VehicleModelView from './VehicleModelView';

const FleetsList = ({ initialSearchQuery = '', rowsPerPage = 10, accountId = null, isEmbedded = false }) => {
  // Always use the base URL that works - '/fleets/list/all/'
  const baseFetchUrl = '/fleets/list/all/';
  
  const [selectedFleet, setSelectedFleet] = useState(null);
  const [selectedVendor, setSelectedVendor] = useState(null);
  const [selectedModel, setSelectedModel] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [isEmpty, setIsEmpty] = useState(false);

  const enums = CommonData();

  const searchFields = [
    { 
      name: 'created_date_from', 
      type: 'date', 
      placeholder: 'From Date',
      maxDate: (values) => values.created_date_to || new Date() 
    },
    { 
      name: 'created_date_to', 
      type: 'date', 
      placeholder: 'To Date',
      minDate: (values) => values.created_date_from || null,
      transformValue: (value) => value ? endOfDay(value) : value
    },
    { name: 'q', type: 'text', placeholder: 'Search fleets...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ]
    },
  ];

  // Helper function to set time to end of day (23:59:59) for "To" date
  const endOfDay = (date) => {
    if (!date) return null;
    const d = new Date(date);
    d.setHours(23, 59, 59, 999);
    return d.toISOString();
  };

  // Helper function to format date for API requests
  const formatDateForAPI = (date) => {
    if (!date) return null;
    // If it's already an ISO string, return as-is
    if (typeof date === 'string' && date.includes('T')) return date;
    // Otherwise, format as ISO string
    return new Date(date).toISOString();
  };

  // Build the complete fetch URL with all parameters
  const buildFetchUrl = (searchParams = {}) => {
    const params = new URLSearchParams();
    
    // Add account_id if provided
    if (accountId) {
      params.append('account_id', accountId);
    }
    
    // Add all search parameters
    Object.keys(searchParams).forEach(key => {
      if (searchParams[key] !== undefined && searchParams[key] !== null && searchParams[key] !== '') {
        params.append(key, searchParams[key]);
      }
    });
    
    // Add pagination parameters
    params.append('skip', searchParams.skip || 0);
    params.append('limit', searchParams.limit || rowsPerPage);
    
    return `${baseFetchUrl}?${params.toString()}`;
  };

  // Transform search parameters before sending to API
  const transformSearchParams = (params) => {
    const transformed = { ...params };
    
    // Format dates for API
    if (transformed.created_date_from) {
      transformed.created_date_from = formatDateForAPI(transformed.created_date_from);
    }
    if (transformed.created_date_to) {
      // Note: created_date_to was already transformed to end of day in searchFields
      transformed.created_date_to = formatDateForAPI(transformed.created_date_to);
    }
    
    // Validate date range
    if (transformed.created_date_from && transformed.created_date_to) {
      const fromDate = new Date(transformed.created_date_from);
      const toDate = new Date(transformed.created_date_to);
      
      if (fromDate > toDate) {
        // Swap dates if they're in wrong order
        [transformed.created_date_from, transformed.created_date_to] = 
          [transformed.created_date_to, transformed.created_date_from];
      }
    }
    
    return transformed;
  };

  // Custom data transformer to check if data is empty
  const transformFleetData = (apiData) => {
    if (!apiData || !apiData.fleets) {
      console.error('❌ Invalid API data structure', apiData);
      setIsEmpty(true);
      return { total_count: 0, fleets: [] };
    }

    setIsEmpty(apiData.fleets.length === 0);
    
    return {
      ...apiData,
      fleets: apiData.fleets.map(flattenFleetData)
    };
  };

  const flattenFleetData = (fleet) => {
    if (!fleet) return {};

    let vehicleColor = 'None';
    let vehicleYear = 'None';
    let vehicleRegistration = 'None';
    let vehicleVendorId = 'None';
    let vehicleVendorName = 'None';
    let vehicleCount = 0;

    if (Array.isArray(fleet.vehicles)) {
      vehicleCount = fleet.vehicles.length;

      if (fleet.vehicles.length > 0) {
        const firstVehicle = fleet.vehicles[0];
        vehicleColor = firstVehicle.color || 'None';
        vehicleYear = firstVehicle.year ? String(firstVehicle.year) : 'None';
        vehicleRegistration = firstVehicle.registration_number || 'None';
        vehicleVendorId = firstVehicle.vendor_id || 'None';
        vehicleVendorName = firstVehicle.vendor_name || 'None';
      }
    }

    return {
      ...fleet,
      vehicle_color: vehicleColor,
      vehicle_year: vehicleYear,
      vehicle_registration: vehicleRegistration,
      vendor_id: vehicleVendorId,
      vendor_name: vehicleVendorName,
      vehicle_count: vehicleCount,
      formatted_date: fleet.created_date 
        ? new Date(fleet.created_date).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
          })
        : 'None',
      status: fleet.status || 'active',
      vehicles: fleet.vehicles || []
    };
  };

  const FLEET_HEADER_TO_KEY_MAP = {
    "Fleet Number": { 
      key: "fleet_name", 
      label: "Fleet Number", 
      sortable: true,
      clickable: true 
    },
    "Fleet Model": { 
      key: "vehicle_type", 
      label: "Fleet Model", 
      sortable: true,
      clickable: true,
      idKey: "vehicle_model_id"
    },
    "Color": { key: "color", label: "Color", sortable: false, clickable: false },
    "Year": { key: "year", label: "Year", sortable: true, clickable: false },
    "Registration": { key: "registration_number", label: "Registration", sortable: true, clickable: false },
    "Vendor": { 
      key: "vendor_name", 
      label: "Vendor", 
      sortable: true,
      clickable: true,
      idKey: "vendor_id"
    },
    "Status": { key: "status", label: "Status", sortable: false, clickable: false },
    "Created Date": { key: "created_date", label: "Created Date", sortable: false, clickable: false },
  };

  const handleActionClick = (fleetId, action) => {
    console.log('🛠️ Action Clicked:', fleetId, action);
    const actionMap = {
      view: 'view',
      edit: 'update',
      delete: 'delete'
    };

    setSelectedFleet({ 
      id: fleetId, 
      modalType: actionMap[action.label],
      action: action.label
    });
  };

  const handleCellClick = (header, item) => {
    const mappingValue = FLEET_HEADER_TO_KEY_MAP[header];
    console.log('🧩 handleCellClick → Header:', header);
    console.log('📦 Clicked Item:', item);
    console.log('🗺️ Mapping Value:', mappingValue);

    if (!mappingValue) {
      console.warn('⚠️ No mapping found for header:', header);
      return;
    }

    switch (mappingValue.key) {
      case 'fleet_name':
        console.log('✅ Fleet Name Clicked → Opening FleetView');
        setSelectedFleet({
          id: item.fleet_id,
          modalType: 'view',
          data: item
        });
        break;

      case 'vehicle_type':
        if (mappingValue.idKey && item[mappingValue.idKey]) {
          console.log('✅ Vehicle Model Clicked → Opening VehicleModelView');
          setSelectedModel({
            id: item[mappingValue.idKey],
            modalType: 'view',
            data: item
          });
        } else {
          console.error('🚫 Missing vehicle model ID in item:', item);
        }
        break;

      case 'vendor_name':
        if (mappingValue.idKey && item[mappingValue.idKey]) {
          console.log('✅ Vendor Clicked → Opening VendorView');
          setSelectedVendor({
            id: item[mappingValue.idKey],
            modalType: 'view',
            data: item
          });
        } else {
          console.error('🚫 Missing vendor ID in item:', item);
        }
        break;

      default:
        console.log('🪵 Cell clicked with no special action:', header);
    }
  };

  const closeModal = () => {
    console.log('🔒 Closing all modals');
    setSelectedFleet(null);
    setSelectedVendor(null);
    setSelectedModel(null);
  };

  const refreshList = () => setRefreshTrigger(prev => prev + 1);

  const actionFormats = [
    { label: "view", modal: true, url: "/fleets/view/{id}" },
    { label: "edit", modal: true, url: "/fleets/update/{id}" },
    { label: "delete", modal: true, url: "/fleets/delete/{id}" }
  ];

  // Custom empty state component
  const EmptyState = () => (
    <div style={{ 
      padding: '40px', 
      textAlign: 'center', 
      color: '#666',
      backgroundColor: '#f8f9fa',
      borderRadius: '8px'
    }}>
      <span
        className="material-symbols-outlined"
        style={{ fontSize: '48px', color: '#9e9e9e', marginBottom: '16px', display: 'block' }}
      >
        local_shipping
      </span>
      <div>No fleet found for this account</div>
    </div>
  );

  return (
    <div className="fleet-list-container">
      {isEmbedded && isEmpty ? (
        <EmptyState />
      ) : (
        <SearchPage
          fetchDataUrl={baseFetchUrl}
          ListComponent={List}
          itemKeyMapping={FLEET_HEADER_TO_KEY_MAP}
          dataKey="fleets"
          identifierKey="fleet_id"
          initialSearchQuery={initialSearchQuery}
          rowsPerPage={rowsPerPage}
          actionFormats={actionFormats}
          onActionClick={handleActionClick}
          onCellClick={handleCellClick}
          refreshTrigger={refreshTrigger}
          dataTransformer={transformFleetData}
          searchFields={searchFields}
          transformSearchParams={transformSearchParams}
          buildFetchUrl={buildFetchUrl} // Add this custom URL builder
          renderCell={(value, column) => {
            if (value === undefined || value === null) {
              return <span style={{ color: '#999' }}>None</span>;
            }
            return String(value);
          }}
          tableProps={{
            striped: true,
            hover: true,
            responsive: true
          }}
          emptyStateComponent={isEmpty && isEmbedded ? EmptyState : null}
        />
      )}

      {selectedFleet?.modalType === "view" && (
        <>
          {console.log('📣 Showing FleetView Modal for ID:', selectedFleet.id)}
          <FleetView 
            fleetId={selectedFleet.id} 
            fleetData={selectedFleet.data}
            onClose={closeModal} 
          />
        </>
      )}

      {selectedFleet?.modalType === "update" && (
        <FleetUpdate 
          fleetId={selectedFleet.id}
          onClose={closeModal}
          onUpdateSuccess={refreshList}
        />
      )}

      {selectedFleet?.modalType === "delete" && (
        <FleetDelete 
          fleetId={selectedFleet.id}
          onClose={closeModal}
          onDeleteSuccess={refreshList}
        />
      )}

      {selectedVendor && (
        <>
          {console.log('📣 Showing VendorView Modal for ID:', selectedVendor.id)}
          <VendorView
            vendorId={selectedVendor.id}
            vendorData={selectedVendor.data}
            onClose={closeModal}
          />
        </>
      )}

      {selectedModel && (
        <>
          {console.log('📣 Showing VehicleModelView Modal for ID:', selectedModel.id)}
          <VehicleModelView
            modelId={selectedModel.id}
            modelData={selectedModel.data}
            onClose={closeModal}
          />
        </>
      )}
    </div>
  );
};

export default FleetsList;