import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import CommonTaskList from '../../common/CommonTaskList';
import TaskBar from '../../common/TaskListBar';


const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'tasks', label: 'Tasks', icon: 'assignment' },
  { id: 'workforce', label: 'Workforce', icon: 'engineering' },
  { id: 'schedules', label: 'Schedules', icon: 'calendar_today' },
  { id: 'alerts', label: 'Alerts', icon: 'warning' },
  { id: 'track', label: 'Track', icon: 'my_location' },
  { id: 'info', label: 'Info', icon: 'info' },
];


const FleetView = ({ fleetId, onClose }) => {
  const [fleetData, setFleetData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);


  useEffect(() => {
    const fetchFleetData = async () => {
      try {
        const response = await axios.get(`/fleets/${fleetId}`);
        setFleetData(response.data);
      } catch (err) {
        setError('Error fetching fleet details.');
      } finally {
        setLoading(false);
      }
    };


    fetchFleetData();
  }, [fleetId]);


  const Icon = ({ name, color = '#000' }) => (
    <span
      className="material-symbols-outlined"
      style={{ fontSize: '24px', color, verticalAlign: 'middle', marginRight: '6px' }}
    >
      {name}
    </span>
  );


  const renderTabContent = () => {
    if (!fleetData) return null;


    switch (activeTab) {
      case 'dashboard':
        return (
          <div style={{ padding: '20px', marginTop: '-20px', marginBottom: '20px' }}>
            <h4><Icon name="dashboard" /> Fleet Summary</h4>
            <p><strong>Name:</strong> {fleetData.fleet_name || 'N/A'}</p>
            <p><strong>Status:</strong> {fleetData.status || 'N/A'}</p>
            <p><strong>Last Activity:</strong> {new Date(fleetData.updated_date).toLocaleString()}</p>
            <TaskBar vehicleId={fleetId} />
            <h4 style={{ marginTop: '20px', marginBottom: '10px' }}>Recent Activity</h4>
            <CommonTaskList presetFilters={{ vehicle_id: fleetId }} rowsPerPage={5} initialSearchQuery="" />
          </div>
        );


      case 'tasks':
        return (
          <div style={{ padding: '20px', marginTop: '-20px' }}>
            <h4><Icon name="assignment" /> Fleet Tasks</h4>
            <CommonTaskList presetFilters={{ vehicle_id: fleetId }} rowsPerPage={5} initialSearchQuery="" />
          </div>
        );


      case 'workforce':
        return (
          <div style={{ padding: '20px', marginTop: '-20px' }}>
            <h4><Icon name="engineering" /> Assigned Workforce</h4>
            {fleetData.assigned_workforce?.length > 0 ? (
              <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fill, minmax(200px, 1fr))', gap: '15px' }}>
                {fleetData.assigned_workforce.map(worker => (
                  <div key={worker.id} style={{ border: '1px solid #ddd', borderRadius: '5px', padding: '10px' }}>
                    <div style={{
                      width: '40px',
                      height: '40px',
                      borderRadius: '50%',
                      backgroundColor: '#e0e0e0',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'center',
                      marginBottom: '10px'
                    }}>
                      {worker.name?.charAt(0)?.toUpperCase() || 'W'}
                    </div>
                    <div>
                      <p><strong>Name:</strong> {worker.name}</p>
                      <p><strong>Role:</strong> {worker.role}</p>
                      <p><strong>Status:</strong> {worker.status}</p>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <p>No workforce currently assigned</p>
            )}
          </div>
        );


      case 'schedules':
        return (
          <div style={{ padding: '20px', marginTop: '-20px' }}>
            <h4><Icon name="calendar_today" /> Maintenance Schedules</h4>
            {fleetData.maintenance_schedules?.length > 0 ? (
              <div>
                {fleetData.maintenance_schedules.map((schedule, index) => (
                  <div key={index} style={{ marginBottom: '15px', paddingBottom: '15px', borderBottom: '1px solid #eee' }}>
                    <p><strong><Icon name="dashboard" /> {schedule.service_type}</strong></p>
                    <p>Next Due: {new Date(schedule.next_due).toLocaleDateString()}</p>
                    <p>Interval: Every {schedule.interval_days} days</p>
                  </div>
                ))}
              </div>
            ) : (
              <p>No maintenance schedules found</p>
            )}
          </div>
        );


      case 'alerts':
        return (
          <div style={{ padding: '20px', marginTop: '-20px' }}>
            <h4><Icon name="warning" /> Active Alerts</h4>
            {fleetData.active_alerts?.length > 0 ? (
              <div>
                {fleetData.active_alerts.map((alert, index) => (
                  <div key={index} style={{
                    marginBottom: '15px',
                    padding: '10px',
                    borderLeft: `4px solid ${alert.severity === 'high' ? '#f44336' : alert.severity === 'medium' ? '#ff9800' : '#4caf50'}`,
                    backgroundColor: '#f9f9f9'
                  }}>
                    <p><strong><Icon name="warning" /> {alert.title}</strong></p>
                    <p>{alert.description}</p>
                    <p style={{ color: '#666', fontSize: '0.9em' }}>
                      {new Date(alert.created_at).toLocaleString()}
                    </p>
                  </div>
                ))}
              </div>
            ) : (
              <p>No active alerts</p>
            )}
          </div>
        );


      case 'track':
        return (
          <div style={{ padding: '20px', marginTop: '-20px' }}>
            <h4><Icon name="my_location" /> Vehicle Tracking</h4>
            {/* <p><strong>Last Location:</strong> {fleetData.last_location || 'Unknown'}</p>
            <p><strong>Last Updated:</strong> {fleetData.location_updated ? new Date(fleetData.location_updated).toLocaleString() : 'N/A'}</p> */}
            <div style={{
              height: '300px',
              backgroundColor: '#e0e0e0',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              marginTop: '15px',
              borderRadius: '5px'
            }}>
              [Map Visualization Placeholder]
            </div>
          </div>
        );


      case 'info':
        return (
          <div style={{ padding: '20px', marginTop: '-20px' }}>
            <div style={{ display: 'grid', gridTemplateColumns: '1fr', gap: '30px' }}>
              <div>
                <h4><Icon name="info" /> Vehicle Details</h4>
                <p><strong>Model:</strong> {fleetData.vehicle_type}</p>
                <p><strong>Registration:</strong> {fleetData.registration_number}</p>
                <p><strong>Color:</strong> {fleetData.color}</p>
                <p><strong>Year:</strong> {fleetData.year}</p>
                <p><strong>VIN:</strong> {fleetData.vin_number}</p>
                <p><strong>Fuel Type:</strong> {fleetData.fuel_type}</p>
              </div>
            </div>
            <div style={{ marginTop: '20px' }}>
              <h4><Icon name="info" /> Vehicle Details</h4>
              <p><strong>Name:</strong> {fleetData.vendor_name}</p>
              {/* <p><strong>Contact:</strong> {fleetData.vendor_contact}</p>
              <p><strong>Contract:</strong> {fleetData.vendor_contract || 'N/A'}</p> */}
            </div>
          </div>
        );


      default:
        return null;
    }
  };


  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading fleet data...</div>
      </Modal>
    );
  }


  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }


  return (
    <Modal onClose={onClose}>
      {/* Header section */}
      <div style={{
        display: 'flex',
        alignItems: 'center',
        padding: '10px',
        justifyContent: 'space-between',
        borderBottom: '1px solid #eee'
      }}>
        <div style={{ display: 'flex', alignItems: 'center' }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold',
            marginRight: '16px'
          }}>
            {fleetData.vehicle_type?.charAt(0)?.toUpperCase() || 'F'}
          </div>
          <div>
            <div style={{ fontWeight: '600', fontSize: '20px', color: 'black' }}>
              {fleetData.vehicle_type || 'Fleet'}
            </div>
            <div style={{ fontSize: '16px', color: 'black', marginTop: '10px', fontWeight: 'bold' }}>
              ID: #{fleetData?.fleets_id ?? 'N/A'}
            </div>
          </div>
        </div>
      </div>


      {/* Divider */}
      <div style={{ height: '1px', width: '99%', backgroundColor: '#ddd', margin: '18px 6px' }} />


      {/* Tabs */}
      <div style={{
        display: 'flex',
        borderBottom: '1px solid #e0e0e0',
        marginBottom: '20px',
        padding: '0 10px'
      }}>
        {TAB_CONFIG.map(tab => (
          <button
            key={tab.id}
            style={{
              padding: '12px 16px',
              border: 'none',
              backgroundColor: 'transparent',
              cursor: 'pointer',
              fontSize: '14px',
              fontWeight: activeTab === tab.id ? '400' : '400',
              color: '#333',
              position: 'relative',
              whiteSpace: 'nowrap',
              minWidth: activeTab === tab.id ? '100px' : '50px',
              textAlign: 'center',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              gap: '8px'
            }}
            onClick={() => setActiveTab(tab.id)}
            onMouseEnter={() => setHoveredTab(tab.id)}
            onMouseLeave={() => setHoveredTab(null)}
          >
            <span
              className="material-symbols-outlined"
              style={{ color: activeTab === tab.id ? '#34c759' : '#000' }}
            >
              {tab.icon}
            </span>
            {(activeTab === tab.id || hoveredTab === tab.id) && (
  <span style={{ color: activeTab === tab.id ? '#34c759' : '#000' }}>
    {tab.label}
  </span>
)}


            {activeTab === tab.id && (
              <div style={{
                position: 'absolute',
                bottom: '-1px',
                left: '0',
                right: '0',
                height: '2px',
                backgroundColor: '#34c759'
              }} />
            )}
          </button>
        ))}
      </div>


      {/* Content */}
<div style={{ 
    maxHeight: 'calc(100vh - 200px)', 
    overflowY: 'auto',
    scrollbarColor: 'white white', // ✅ make scrollbar track & thumb both white
    scrollbarWidth: 'thin'          // ✅ make scrollbar thin (Firefox)
}}>
    {renderTabContent()}
</div>
    </Modal>
  );
};


export default FleetView;