import React, { useState } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';

// Fleet Schema Definition
export const fleetSchema = {
  title: "Fleet Information",
  type: "object",
  required: ["fleet_name"],
  properties: {
    fleet_name: { 
      type: "string", 
      title: "Fleet Name",
      minLength: 2,
      maxLength: 50 
    },
    description: {
      type: "string",
      title: "Description",
      maxLength: 200
    },
    status: {
      type: "string",
      title: "Status",
      enum: ["active", "inactive", "maintenance"],
      default: "active"
    },
    vehicle_id: {
      type: "string",
      title: "Vehicle ID"
    },
    registration_number: {
      type: "string",
      title: "Registration Number"
    },
    vin_number: {
      type: "string",
      title: "VIN Number"
    },
    color: {
      type: "string",
      title: "Color"
    },
    year: {
      type: "integer",
      title: "Year"
    },
    vehicle_image: {
      type: "string",
      title: "Vehicle Image",
      format: "data-url"
    },
    vendor_id: {
      type: "string",
      title: "Vendor ID"
    }
  }
};

// UI Schema Configuration
export const fleetUiSchema = {
  "ui:order": [
    "fleet_name",
    "description",
    "status",
    "vehicle_id",
    "registration_number",
    "vin_number",
    "color",
    "year",
    "vehicle_image",
    "vendor_id"
  ],
  description: {
    "ui:widget": "textarea",
    "ui:options": {
      rows: 3
    }
  },
  status: {
    "ui:widget": "select"
  },
  vehicle_image: {
    "ui:widget": "fileWidget",
    "ui:options": {
      accept: "image/*"
    }
  }
};

// Custom File Upload Widget
const CustomFileWidget = ({ value, onChange }) => {
  const handleChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        onChange(event.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  return (
    <div style={{ margin: '10px 0' }}>
      {value && (
        <div style={{ marginBottom: '10px' }}>
          <img 
            src={value} 
            alt="Preview" 
            style={{ maxWidth: '150px', maxHeight: '150px', borderRadius: '4px', border: '1px solid #ddd' }}
          />
        </div>
      )}
      <input 
        type="file" 
        onChange={handleChange} 
        accept="image/*"
        style={{ display: 'block' }}
      />
      <p style={{ fontSize: '0.8rem', color: '#666', marginTop: '5px' }}>
        Upload a clear vehicle image
      </p>
    </div>
  );
};

// Main FleetForm Component
const FleetForm = ({ initialData = {}, onSubmit, onCancel }) => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [isHovered, setIsHovered] = useState(false);

  const handleSubmit = async ({ formData }) => {
    setLoading(true);
    try {
      await onSubmit(formData);
    } catch (err) {
      setError(err.response?.data?.message || "Failed to submit");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="fleet-form-container" style={{ maxWidth: '800px', margin: '0 auto' }}>
      {error && (
        <div className="alert alert-danger" style={{ marginBottom: '20px' }}>
          {error}
        </div>
      )}
      
      <Form
        schema={fleetSchema}
        uiSchema={fleetUiSchema}
        formData={{
          status: "active",
          ...initialData
        }}
        onSubmit={handleSubmit}
        validator={validator}
        disabled={loading}
        widgets={{ fileWidget: CustomFileWidget }}
      >
        <div className="form-actions" style={{ 
          display: 'flex', 
          justifyContent: 'flex-start', 
          gap: '10px', 
          marginTop: '20px' 
        }}>
          {onCancel && (
            <button 
              type="button" 
              onClick={onCancel}
              className="btn btn-secondary"
              disabled={loading}
              style={{
                padding: '8px 16px',
                borderRadius: '4px',
                border: '1px solid #6c757d',
                backgroundColor: '#6c757d',
                color: '#f0f0f0',
                cursor: 'pointer'
              }}
            >
              Cancel
            </button>
          )}
<button 
  type="submit"
  style={{
    backgroundColor: '#f0f0f0', // pale white background
    color: '#00D563',           // green text
    border: '1px solid white',
    padding: '10px 20px',
    borderRadius: '6px',
    fontSize: '14px',
    fontWeight: 'bold',
    cursor: 'pointer',
    marginTop: '20px',
    textAlign: 'left' 
              
  }}
  onMouseEnter={e => {
    e.target.style.backgroundColor = '#00D563';
    e.target.style.color = '#f0f0f0';
  }}
  onMouseLeave={e => {
    e.target.style.backgroundColor = '#f0f0f0';
    e.target.style.color = '#00D563';
  }}
>
  Submit
</button>
        </div>
      </Form>
    </div>
  );
};

export default FleetForm;