import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import { useNavigate } from 'react-router-dom'; // Add this import

const dataURLtoFile = (dataurl, filename) => {
  const arr = dataurl.split(',');
  const mime = arr[0].match(/:(.*?);/)[1];
  const bstr = atob(arr[1]);
  let n = bstr.length;
  const u8arr = new Uint8Array(n);
  while (n--) {
    u8arr[n] = bstr.charCodeAt(n);
  }
  return new File([u8arr], filename, { type: mime });
};

const AddFleet = ({ onClose = () => {}, onSuccess = () => {} }) => {
  const navigate = useNavigate(); // Initialize navigate
  const [availableVehicles, setAvailableVehicles] = useState([]);
  const [availableVendors, setAvailableVendors] = useState([]);
  const [loading, setLoading] = useState({ 
    models: true, 
    vendors: true,
    submit: false 
  });
  const [error, setError] = useState(null);
  const [successMessage, setSuccessMessage] = useState(null);
  const [selectedVehicleId, setSelectedVehicleId] = useState('');
  const [selectedVehicleModelName, setSelectedVehicleModelName] = useState('');
  const [selectedVendorId, setSelectedVendorId] = useState('');
  const [selectedVendorName, setSelectedVendorName] = useState('');

  const [formData, setFormData] = useState({
    fleet_name: '',
    description: '',
    status: 'active',
    vehicle_id: '',
    vehicle_model_id: '',
    vehicle_model_name: '',
    registration_number: '',
    vin_number: '',
    color: 'white',
    year: new Date().getFullYear(),
    vehicle_image: '',
    vendor_id: '',
    dealer_id: '',
    capacity: '',
    fuel_type: '',
    vehicle_type: '',
    manufacturer_name: ''
  });

  // Fetch vehicle models
  useEffect(() => {
    let isMounted = true;
    const controller = new AbortController();

    const fetchVehicleModels = async () => {
      try {
        setLoading(prev => ({ ...prev, models: true }));
        setError(null);
        
        const response = await axios.get('/vehicle-models/list/all/', {
          signal: controller.signal
        });

        if (!isMounted) return;

        const models = response.data?.vehicles || response.data?.data || 
                      (Array.isArray(response.data) ? response.data : []);

        if (!models || models.length === 0) {
          throw new Error('No vehicle models available.');
        }

        setAvailableVehicles(models);
      } catch (err) {
        if (!axios.isCancel(err)) {
          setError(err.response?.data?.message || 'Failed to load vehicle models.');
        }
      } finally {
        if (isMounted) {
          setLoading(prev => ({ ...prev, models: false }));
        }
      }
    };

    fetchVehicleModels();

    return () => {
      isMounted = false;
      controller.abort();
    };
  }, []);

  // Fetch vendors
  useEffect(() => {
    let isMounted = true;
    const controller = new AbortController();

    const fetchVendors = async () => {
      try {
        setLoading(prev => ({ ...prev, vendors: true }));
        setError(null);
        
        const response = await axios.get('/vendors/list/all/', {
          signal: controller.signal
        });

        if (!isMounted) return;

        const vendors = response.data?.vendors || response.data?.data || 
                       (Array.isArray(response.data) ? response.data : []);

        if (!vendors || vendors.length === 0) {
          console.log('No vendors available');
        }

        setAvailableVendors(vendors);
      } catch (err) {
        if (!axios.isCancel(err)) {
          console.error('Failed to load vendors:', err);
        }
      } finally {
        if (isMounted) {
          setLoading(prev => ({ ...prev, vendors: false }));
        }
      }
    };

    fetchVendors();

    return () => {
      isMounted = false;
      controller.abort();
    };
  }, []);

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      setError(null);
      setSuccessMessage(null);
      setLoading(prev => ({ ...prev, submit: true }));
  
      if (!selectedVehicleId) {
        throw new Error('Please select a vehicle model.');
      }

      if (!formData.fleet_name.trim()) {
        throw new Error('Fleet name is required.');
      }
  
      const selectedModel = availableVehicles.find(
        model => String(model._id) === String(selectedVehicleId) || 
                String(model.vehicle_id) === String(selectedVehicleId)
      );
  
      if (!selectedModel) {
        throw new Error('Selected vehicle model not found.');
      }
  
      const payload = {
        fleet_name: formData.fleet_name,
        description: formData.description,
        status: formData.status,
        vehicle_id: selectedVehicleId,
        vehicle_model_id: selectedVehicleId,
        vehicle_model_name: selectedVehicleModelName,
        registration_number: formData.registration_number,
        vin_number: formData.vin_number,
        color: formData.color,
        year: formData.year,
        vehicle_image: formData.vehicle_image,
        vendor_id: selectedVendorId,
        vendor_name: selectedVendorName,
        dealer_id: formData.dealer_id,
        capacity: formData.capacity,
        fuel_type: selectedModel.fuel_type,
        vehicle_type: selectedModel.vehicle_type,
        manufacturer_name: selectedModel.manufacturer_name
      };
  
      const formDataToSend = new FormData();
        Object.entries(payload).forEach(([key, value]) => {
          if (key !== 'vehicle_image') {
            formDataToSend.append(key, value);
          }
        });

        if (payload.vehicle_image?.startsWith('data:')) {
          const imageFile = dataURLtoFile(payload.vehicle_image, `fleet-${Date.now()}.png`);
          formDataToSend.append('vehicle_image', imageFile);
        }

        const response = await axios.post('/fleets/', formDataToSend, {
          headers: { 'Content-Type': 'multipart/form-data' }
        });

  
      if (response.status === 201) {
        const fleetData = response.data;
        window.alert('Fleet created successfully!');
        navigate('/fleets'); // Redirect to fleets list after alert is dismissed
        onSuccess(fleetData);
        onClose();
        return;
      }
  
      throw new Error('Fleet creation failed.');
    } catch (err) {
      setError(err.response?.data?.message || err.message || 'Failed to create fleet.');
    } finally {
      setLoading(prev => ({ ...prev, submit: false }));
    }
  };

  const handleVehicleSelect = (e) => {
    const vehicleId = e.target.value;
    const selectedModel = availableVehicles.find(
      model => String(model._id) === String(vehicleId) || 
              String(model.vehicle_id) === String(vehicleId)
    );

    if (selectedModel) {
      setSelectedVehicleId(vehicleId);
      setSelectedVehicleModelName(selectedModel.manufacturer_name);
      setFormData(prev => ({
        ...prev,
        vehicle_id: vehicleId,
        vehicle_model_id: vehicleId,
        vehicle_model_name: selectedModel.manufacturer_name,
        fuel_type: selectedModel.fuel_type,
        vehicle_type: selectedModel.vehicle_type,
        manufacturer_name: selectedModel.manufacturer_name
      }));
    }
  };

  const handleVendorSelect = (e) => {
    const vendorId = e.target.value;
    const selectedVendor = availableVendors.find(
      vendor => String(vendor._id || vendor.vendor_id) === vendorId
    );

    if (selectedVendor) {
      setSelectedVendorId(vendorId);
      setSelectedVendorName(selectedVendor.vendor_name || selectedVendor.name || '');
      
      setFormData(prev => ({
        ...prev,
        vendor_id: vendorId
      }));
    }
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  return (
    <div style={{ maxWidth: '600px', padding: '20px' }}>
      {loading.models ? (
        <div className="text-center p-4">
          <p>Loading vehicle models...</p>
        </div>
      ) : error ? (
        <div className="alert alert-danger">{error}</div>
      ) : successMessage ? (
        <div className="alert alert-success">{successMessage}</div>
      ) : (
        <form onSubmit={handleSubmit}>
          <div className="mb-3">
            <label className="form-label">Fleet Model *</label>
            <select
              className="form-select"
              value={selectedVehicleId}
              onChange={handleVehicleSelect}
              required
              disabled={loading.submit}
            >
              <option value="">-- Select a model --</option>
              {availableVehicles.map(model => (
                <option 
                  key={model._id || model.vehicle_id} 
                  value={model._id || model.vehicle_id}
                >
                  {model.vehicle_type}
                </option>
              ))}
            </select>
          </div>

          <div className="mb-3">
            <label className="form-label">Fleet Number *</label>
            <input
              type="text"
              name="fleet_name"
              value={formData.fleet_name}
              onChange={handleInputChange}
              required
              className="form-control"
              placeholder="e.g., TN12345"
              disabled={loading.submit}
            />
          </div>

          <div className="mb-3">
            <label className="form-label">Registration Number</label>
            <input
              type="text"
              name="registration_number"
              value={formData.registration_number}
              onChange={handleInputChange}
              className="form-control"
              placeholder="Enter registration number"
              disabled={loading.submit}
            />
          </div>

          <div className="mb-3">
            <label className="form-label">VIN Number</label>
            <input
              type="text"
              name="vin_number"
              value={formData.vin_number}
              onChange={handleInputChange}
              required
              className="form-control"
              placeholder="17-character VIN"
              minLength={17}
              maxLength={17}
              disabled={loading.submit}
            />
          </div>

          <div className="mb-3">
            <label className="form-label">Color</label>
            <input
              type="text"
              name="color"
              value={formData.color}
              onChange={handleInputChange}
              required
              className="form-control"
              placeholder="e.g., Red"
              disabled={loading.submit}
            />
          </div>

          <div className="mb-3">
            <label className="form-label">Year</label>
            <input
              type="number"
              name="year"
              value={formData.year}
              onChange={handleInputChange}
              min="1900"
              max={new Date().getFullYear() + 1}
              required
              className="form-control"
              disabled={loading.submit}
            />
          </div>

          <div className="mb-3">
            <label className="form-label">Vehicle Image Upload</label>
            <input
              type="file"
              accept="image/*"
              className="form-control"
              onChange={(e) => {
                const file = e.target.files[0];
                if (file) {
                  const reader = new FileReader();
                  reader.onloadend = () => {
                    setFormData(prev => ({
                      ...prev,
                      vehicle_image: reader.result
                    }));
                  };
                  reader.readAsDataURL(file);
                }
              }}
            />
            {formData.vehicle_image && (
              <div className="mt-2">
                <img 
                  src={formData.vehicle_image} 
                  alt="Preview" 
                  style={{ maxWidth: '150px', borderRadius: '4px', border: '1px solid #ccc' }} 
                />
              </div>
            )}
          </div>

          <div className="mb-3">
            <label className="form-label">Vendor *</label>
            <select
              className="form-select"
              value={selectedVendorId}
              onChange={handleVendorSelect}
              required
              disabled={loading.submit}
            >
              <option value="">-- Select a vendor --</option>
              {availableVendors.map(vendor => (
                <option 
                  key={vendor._id || vendor.vendor_id} 
                  value={vendor._id || vendor.vendor_id}
                >
                  {vendor.vendor_name || vendor.name}
                </option>
              ))}
            </select>
          </div>

          <div className="mb-4">
            <label className="form-label">Dealer ID</label>
            <input
              type="text"
              name="dealer_id"
              value={formData.dealer_id}
              onChange={handleInputChange}
              required
              className="form-control"
              placeholder="Enter dealer ID"
              disabled={loading.submit}
            />
          </div>

          <div className="d-flex justify-content-end mt-5">
            <style>{`
              .btn-shared {
                background-color:'white;
                color: #333;
                border: none;
                padding: 8px 16px;
                font-weight: bold;
                border-radius: 4px;
                transition: all 0.2s ease;
                margin-left: 10px;
              }

              .btn-shared:hover {
                background-color: #28a745;
                color: white;
              }

              .btn-shared:disabled {
                opacity: 0.6;
                cursor: not-allowed;
              }
            `}</style>

            <button
              type="submit"
              className="btn-shared"
              disabled={loading.submit || !selectedVehicleId}
              style={{ 
                marginTop: '20px',
                marginLeft: '4px',
                padding: '9px 15px'
              }}
            >
              {loading.submit ? 'Creating...' : 'Submit'}
            </button>
          </div>
        </form>
      )}
    </div>
  );
};

export default AddFleet;