import React, { useEffect, useState, useRef } from 'react';
import axios from 'axios';
import GoogleMapReact from 'google-map-react';
import CloudUploadIcon from '@mui/icons-material/CloudUpload';
import AddIcon from '@mui/icons-material/Add';
import ScheduleIcon from '@mui/icons-material/Schedule';
import ChevronLeftIcon from '@mui/icons-material/ChevronLeft';
import ChevronRightIcon from '@mui/icons-material/ChevronRight';
import Script from 'react-load-script';
import Tabs from '../../common/Tabs';
import '../../../Dispatch.css';



const DispatchPage = ({ defaultActiveTab = 'schedules', hideSidebarTabs = false }) => {
  // State declarations
  const [services, setServices] = useState([]);
  const [subServices, setSubServices] = useState([]);
  const [selectedService, setSelectedService] = useState(null);
  const [selectedSubService, setSelectedSubService] = useState(null);
  const [tasks, setTasks] = useState([]);
  const [rightSidebarOpen, setRightSidebarOpen] = useState(false);
  const [activeTab, setActiveTab] = useState(defaultActiveTab);
  const [newTaskFormOpen, setNewTaskFormOpen] = useState(false);
  const [uploadSidebarOpen, setUploadSidebarOpen] = useState(false);
  const [uploading, setUploading] = useState(false);

  const [mapCenter] = useState({ lat: 40.7128, lng: -74.0060 });
  const [mapZoom] = useState(12);
  const [serviceSearchTerm, setServiceSearchTerm] = useState('');
  const [subServiceSearchTerm, setSubServiceSearchTerm] = useState('');
  const [workforceData, setWorkforceData] = useState([]);
  const [loadingWorkforce, setLoadingWorkforce] = useState(false);
  const [workforceSelection, setWorkforceSelection] = useState('automatic');
  const [googleMapsReady, setGoogleMapsReady] = useState(false);
  const [customerOptions, setCustomerOptions] = useState([]);
  const [showCustomerDropdown, setShowCustomerDropdown] = useState(false);
  const [existingSchedules, setExistingSchedules] = useState([]);
  const [loadingSchedules, setLoadingSchedules] = useState(false);
  const [scheduleType, setScheduleType] = useState('new');
  const [selectedSchedule, setSelectedSchedule] = useState(null);
  const [scheduleSearchTerm, setScheduleSearchTerm] = useState('');
  const [selectedWorkforce, setSelectedWorkforce] = useState(null);
  const [isDispatching, setIsDispatching] = useState(false);
  const [dispatchSuccess, setDispatchSuccess] = useState(null);
  const [customers, setCustomers] = useState([]);
  const [loadingCustomers, setLoadingCustomers] = useState(false);
  const [servicePage, setServicePage] = useState(0);
  const servicesPerPage = 3;
  const [subServicePage, setSubServicePage] = useState(0);
  const subServicesPerPage = 3;
  const [fileUploaded, setFileUploaded] = useState(false);
  const [mapsLoading, setMapsLoading] = useState(true);
  const [allocationType, setAllocationType] = useState('individual');
  const [showCombinedSidebar, setShowCombinedSidebar] = useState(false);
  const [sidebarContent, setSidebarContent] = useState('schedules');
  const [workforceSearchTerm, setWorkforceSearchTerm] = useState('');
  const [scheduleName, setScheduleName] = useState('');
  const [selectedWorkforceDetails, setSelectedWorkforceDetails] = useState(null);
  const [loadingWorkforceDetails, setLoadingWorkforceDetails] = useState(false);
  const [createdScheduleId, setCreatedScheduleId] = useState(null);
  const [hoveredWorkerId, setHoveredWorkerId] = useState(null);

  // NEW: inline form validation + UI banner message
  const [formErrors, setFormErrors] = useState({});
  const [uiMessage, setUiMessage] = useState(null); // { type: 'success' | 'error', text: string }

  // Refs
  const pickupAutocompleteRef = useRef(null);
  const dropAutocompleteRef = useRef(null);
  const pickupInputRef = useRef(null);
  const dropInputRef = useRef(null);
  const customerInputRef = useRef(null);
  const mapsScriptLoaded = useRef(false);

  const [taskFormData, setTaskFormData] = useState({
    customer: '',
    customerId: '',
    location1: '',
    location2: '',
    dateTime: '',
    location1Coords: null,
    location2Coords: null
  });

  // Filtered data
  const filteredServices = services.filter(service =>
    service.service_title?.toLowerCase().includes(serviceSearchTerm.toLowerCase())
  );

  const filteredSubServices = subServices.filter(subService =>
    subService.service_title?.toLowerCase().includes(subServiceSearchTerm.toLowerCase())
  );

  const filteredSchedules = existingSchedules.filter(schedule =>
    schedule.schedule_name?.toLowerCase().includes(scheduleSearchTerm.toLowerCase())
  );

  const visibleServices = filteredServices.slice(
    servicePage * servicesPerPage,
    servicePage * servicesPerPage + servicesPerPage
  );

  const visibleSubServices = filteredSubServices.slice(
    subServicePage * subServicesPerPage,
    subServicePage * subServicesPerPage + subServicesPerPage
  );

  const filteredWorkforce = workforceData.filter(worker =>
    worker.first_name?.toLowerCase().includes(workforceSearchTerm.toLowerCase())
  );

  // Navigation handlers
  const handleNext = () => {
    if ((servicePage + 1) * servicesPerPage < filteredServices.length) {
      setServicePage(servicePage + 1);
    }
  };

  const handlePrev = () => {
    if (servicePage > 0) {
      setServicePage(servicePage - 1);
    }
  };

  const handleSubServiceNext = () => {
    if ((subServicePage + 1) * subServicesPerPage < filteredSubServices.length) {
      setSubServicePage(subServicePage + 1);
    }
  };

  const handleSubServicePrev = () => {
    if (subServicePage > 0) {
      setSubServicePage(subServicePage - 1);
    }
  };

  const handleScheduleTypeChange = (type) => {
    setScheduleType(type);
    setScheduleSearchTerm('');
    if (type === 'new') {
      setSelectedSchedule(null);
    }
  };

  // === VALIDATE + CREATE/OPEN ON "GO" ===
  // === VALIDATE + CREATE/OPEN ON "GO" ===
const handleGoClick = async () => {
  setUiMessage(null);

  // If no uploaded tasks, require the single inline form fields
  const needsInline = tasks.length === 0;
  const errors = {};

  if (needsInline) {
    if (!taskFormData.customer) errors.customer = "Customer is required";
    if (!taskFormData.location1) errors.location1 = "Pickup location is required";
    if (!taskFormData.location2) errors.location2 = "Drop location is required";
    if (!taskFormData.dateTime) errors.dateTime = "Date & time is required";
  }

  setFormErrors(errors);
  if (Object.keys(errors).length > 0) return;

  try {
    const accountId = localStorage.getItem('account_id') || '67fec97f95809e100c9954d4';

    if (scheduleType === 'new') {
      // one-line inline stop (pickup + drop in same object)
      const inlineStop = (needsInline || taskFormData.customer)
        ? [{
            customer_name: taskFormData.customer,
            customer_id: taskFormData.customerId,
            location: taskFormData.location1,        // pickup
            drop_location: taskFormData.location2,   // drop
            type: "",
            sequence: "1",
            timing: taskFormData.dateTime || ""
          }]
        : [];

      const payload = {
        account_id: accountId,
        description: "",
        start_time: needsInline
          ? taskFormData.dateTime
          : (tasks[0]?.dateTime || new Date().toISOString()),
        end_time: null,
        task_stops: [
          // uploaded tasks → include both pickup & drop in same object
          ...tasks.map((task) => ({
            customer_name: task.customer,
            customer_id: task.customerId,
            location: task.location1,        // pickup
            drop_location: task.location2,   // drop
            type: "",
            sequence: "1",
            timing: task.dateTime || ""
          })),
          ...inlineStop
        ]
      };

      const response = await axios.post(
        'schedules/create-with-tasks',
        payload
      );

      if (response.status === 200 && !response.data.error) {
        setUiMessage({ type: 'success', text: 'New schedule created successfully.' });
        setCreatedScheduleId(response.data.schedule_id);

        if (workforceSelection === 'manual') {
          await fetchWorkforceData();
        }

        setShowCombinedSidebar(true);
      } else {
        throw new Error(response.data?.message || 'Unknown error occurred while creating schedule');
      }
    } else if (scheduleType === 'existing') {
      if (workforceSelection === 'manual') {
        await fetchWorkforceData();
      }
      setShowCombinedSidebar(true);
    }
  } catch (error) {
    console.error('Error in handleGoClick:', error);
    const errorMessage =
      error.response?.data?.message ||
      error.response?.data?.detail ||
      error.message ||
      'Something went wrong while creating the schedule';
    setUiMessage({ type: 'error', text: `Failed: ${errorMessage}` });
  }
};


  // Hide page header
  useEffect(() => {
    const header = document.querySelector('.PageHeader');
    if (header) header.style.display = 'none';
    return () => {
      if (header) header.style.display = '';
    };
  }, []);

  // Load customers
  useEffect(() => {
    const fetchCustomers = async () => {
      try {
        setLoadingCustomers(true);
        const accountId = localStorage.getItem('account_id') || '67fec97f95809e100c9954d4';
        const response = await axios.get(
          `customers/?account_id=${accountId}&skip=0&limit=10&q=&created_date_from=&created_date_to=&customer_type=&status=`
        );
        setCustomers(response.data?.users || []);
      } catch (error) {
        console.error("Error fetching customers:", error);
      } finally {
        setLoadingCustomers(false);
      }
    };

    fetchCustomers();
  }, []);

  // Initialize Google Maps autocomplete
  useEffect(() => {
    if (!googleMapsReady || !newTaskFormOpen) return;

    const initAutocomplete = () => {
      if (pickupInputRef.current && !pickupAutocompleteRef.current) {
        pickupAutocompleteRef.current = new window.google.maps.places.Autocomplete(
          pickupInputRef.current,
          { 
            types: ['geocode'],
            fields: ['formatted_address', 'geometry'],
            componentRestrictions: { country: 'in' }
          }
        );

        pickupAutocompleteRef.current.addListener('place_changed', () => {
          const place = pickupAutocompleteRef.current.getPlace();
          if (place.geometry) {
            setTaskFormData(prev => ({
              ...prev,
              location1: place.formatted_address,
              location1Coords: place.geometry.location.toJSON()
            }));
            setFormErrors(prev => ({ ...prev, location1: undefined }));
          }
        });
      }

      if (dropInputRef.current && !dropAutocompleteRef.current) {
        dropAutocompleteRef.current = new window.google.maps.places.Autocomplete(
          dropInputRef.current,
          { 
            types: ['geocode'],
            fields: ['formatted_address', 'geometry'],
            componentRestrictions: { country: 'in' }
          }
        );

        dropAutocompleteRef.current.addListener('place_changed', () => {
          const place = dropAutocompleteRef.current.getPlace();
          if (place.geometry) {
            setTaskFormData(prev => ({
              ...prev,
              location2: place.formatted_address,
              location2Coords: place.geometry.location.toJSON()
            }));
            setFormErrors(prev => ({ ...prev, location2: undefined }));
          }
        });
      }
    };

    if (window.google && window.google.maps && window.google.maps.places) {
      initAutocomplete();
    } else {
      const checkInterval = setInterval(() => {
        if (window.google && window.google.maps && window.google.maps.places) {
          initAutocomplete();
          clearInterval(checkInterval);
        }
      }, 100);
    }

    return () => {
      if (pickupAutocompleteRef.current) {
        window.google.maps.event.clearInstanceListeners(pickupAutocompleteRef.current);
        pickupAutocompleteRef.current = null;
      }
      if (dropAutocompleteRef.current) {
        window.google.maps.event.clearInstanceListeners(dropAutocompleteRef.current);
        dropAutocompleteRef.current = null;
      }
    };
  }, [googleMapsReady, newTaskFormOpen]);

  // Google Maps Script loading
  const handleScriptLoad = () => {
    setGoogleMapsReady(true);
    setMapsLoading(false);
    mapsScriptLoaded.current = true;
  };

  const handleScriptError = () => {
    console.error("Failed to load Google Maps API");
    setGoogleMapsReady(false);
    setMapsLoading(false);
  };

  // Load services
  useEffect(() => {
    axios.get('services/getsubservicelist/67fec97f95809e100c9954d4?&is_parent=true')
      .then((res) => {
        const servicesData = res.data?.users || [];
        setServices(servicesData);
        if (servicesData.length > 0) {
          setSelectedService(servicesData[0]);
        }
      })
      .catch(console.error);
  }, []);

  // Load sub-services
  useEffect(() => {
    if (selectedService?.id) {
      axios.get(`services/getsubservicelist/67fec97f95809e100c9954d4?service_id=${selectedService.id}&is_parent=false`)
        .then((res) => {
          const subServicesData = res.data?.users || [];
          setSubServices(subServicesData);
          if (subServicesData.length > 0) {
            setSelectedSubService(subServicesData[0]);
          }
        })
        .catch(console.error);
    } else {
      setSubServices([]);
      setSelectedSubService(null);
    }
  }, [selectedService]);

  // Load existing schedules
  useEffect(() => {
    if (scheduleType === 'existing') {
      const timer = setTimeout(() => {
        fetchExistingSchedules();
      }, 500);
      return () => clearTimeout(timer);
    } else {
      setExistingSchedules([]);
      setSelectedSchedule(null);
    }
  }, [scheduleType, scheduleSearchTerm]);

  const fetchExistingSchedules = async () => {
    setLoadingSchedules(true);
    try {
      const accountId = localStorage.getItem('account_id') || '67fec97f95809e100c9954d4';
      const response = await axios.get(
        `schedules/search?account_id=${accountId}&q=${scheduleSearchTerm}`
      );
      setExistingSchedules(response.data || []);
    } catch (error) {
      console.error('Error fetching schedules:', error);
      setExistingSchedules([]);
    } finally {
           setLoadingSchedules(false);
    }
  };

  const fetchWorkforceData = async () => {
    setLoadingWorkforce(true);
    try {
      const accountId = localStorage.getItem('account_id') || '67fec97f95809e100c9954d4';
      const typeParam = scheduleType === 'new' ? 'free' : 'scheduled';
      const url = `workforce/freeworkforce/${accountId}?type=${typeParam}`;
      const response = await axios.get(url);
      setWorkforceData(response.data || []);
    } catch (err) {
      console.error('Workforce fetch failed', err);
      setWorkforceData([]);
    } finally {
      setLoadingWorkforce(false);
    }
  };

  const fetchWorkforceDetails = async (workforceId) => {
    setLoadingWorkforceDetails(true);
    try {
      const response = await axios.post(
        "workforce/details",
        { workforce_id: workforceId },
        { headers: { "Content-Type": "application/json" } }
      );

      if (response.data && !response.data.error) {
        const details = Array.isArray(response.data) ? response.data[0] : response.data;
        setSelectedWorkforceDetails(details);
      } else {
        throw new Error(response.data?.message || "Failed to fetch workforce details");
      }
    } catch (error) {
      console.error("Error fetching workforce details:", error);
      setSelectedWorkforceDetails(null);
    } finally {
      setLoadingWorkforceDetails(false);
    }
  };

  const handleWorkforceSelect = (worker) => {
    setSelectedWorkforce(worker);
    if (worker && worker.id) {
      fetchWorkforceDetails(worker.id);
    }
  };

  const handleSettingsClick = (worker) => {
    setSelectedWorkforce(worker);
    setRightSidebarOpen(true);
    fetchWorkforceDetails(worker.id);
  };

const handleFileUpload = async (event) => {
  const file = event.target.files?.[0];
  if (!file) return;

  // simple validation
  const okType = /\.(xls|xlsx)$/i.test(file.name);
  const okSize = file.size <= 5 * 1024 * 1024; // 5 MB
  if (!okType || !okSize) {
    setUiMessage({ type: 'error', text: 'Please upload a .xls/.xlsx file up to 5MB.' });
    event.target.value = '';
    return;
  }

  try {
    setUploading(true);
    setUiMessage(null);

    const formData = new FormData();
    formData.append("file", file);

    // upload to your backend
    const resp = await axios.post("/api/upload-tasks", formData, {
      headers: { "Content-Type": "multipart/form-data" },
    });

    if (resp.status !== 200) throw new Error('Upload failed');

    // pull parsed tasks AFTER a successful upload
    const tasksResponse = await axios.get("/api/tasks");
    const list = Array.isArray(tasksResponse.data) ? tasksResponse.data : [];
    setTasks(list);

    setFileUploaded(true); // mark success only here
    setUiMessage({ type: 'success', text: `Uploaded. ${list.length} task(s) found.` });
    setUploadSidebarOpen(false);
  } catch (e) {
    console.error("Upload failed:", e);
    setFileUploaded(false);
    setUiMessage({ type: 'error', text: `Upload failed: ${e?.message || 'Unknown error'}` });
  } finally {
    setUploading(false);
    event.target.value = ''; // reset chooser
  }
};


  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setTaskFormData(prev => ({ ...prev, [name]: value }));
    // clear field-specific error while typing
    setFormErrors(prev => ({ ...prev, [name]: undefined }));
  };

  const handleDispatch = async () => {
    if (isDispatching) return;

    setIsDispatching(true);
    setDispatchSuccess(null);
    setUiMessage(null);

    try {
      const accountId = localStorage.getItem('account_id') || '67fec97f95809e100c9954d4';

      const payload = {
        account_id: accountId,
        workforce_id: selectedWorkforce?.id,
        schedule_id:
          scheduleType === 'existing'
            ? (selectedSchedule?.schedule_id || createdScheduleId)
            : createdScheduleId
      };

      if (!payload.schedule_id) {
        throw new Error('Please select or create a schedule before dispatching');
      }

      if (workforceSelection === 'manual' && !payload.workforce_id) {
        throw new Error('Please select a workforce');
      }

      const response = await axios.post(
        'schedules/assign',
        payload
      );

      if (
        response.status === 200 &&
        (response.data.success || response.data.message === "Schedule assigned successfully")
      ) {
        setDispatchSuccess(true);
        setUiMessage({ type: 'success', text: 'Dispatch successful.' });
        setScheduleType('existing');
        setWorkforceSelection('manual');
        setScheduleSearchTerm('');
      } else {
        setDispatchSuccess(false);
        throw new Error(response.data?.message || 'Unknown error');
      }
    } catch (error) {
      console.error('Dispatch API error:', error);
      setDispatchSuccess(false);
      setUiMessage({ type: 'error', text: `Dispatch failed: ${error.message || 'Something went wrong while dispatching.'}` });
    } finally {
      setIsDispatching(false);
    }
  };

  const handleCustomerSearch = async (e) => {
    const searchTerm = e.target.value;
    setTaskFormData(prev => ({ ...prev, customer: searchTerm }));
    setFormErrors(prev => ({ ...prev, customer: undefined }));

    if (!searchTerm || searchTerm.length < 2) {
      setCustomerOptions([]);
      setShowCustomerDropdown(false);
      return;
    }

    try {
      const accountId = localStorage.getItem('account_id') || '67fec97f95809e100c9954d4';
      const response = await axios.get(
        `customers/?account_id=${accountId}&skip=0&limit=10&q=${searchTerm}`
      );
      setCustomerOptions(response.data?.users || []);
      setShowCustomerDropdown(true);
    } catch (error) {
      console.error('Customer search failed:', error);
      setCustomerOptions([]);
      setShowCustomerDropdown(false);
    }
  };

  const handleCustomerSelect = (customer) => {
    setTaskFormData(prev => ({
      ...prev,
      customer: `${customer.first_name} ${customer.last_name}`,
      customerId: customer.id
    }));
    setShowCustomerDropdown(false);
    setFormErrors(prev => ({ ...prev, customer: undefined }));
    if (pickupInputRef.current) {
      pickupInputRef.current.focus();
    }
  };

  const handleMapClick = ({ lat, lng }) => {
    // map click hook if needed
  };

  const handleWorkforceSelectionChange = (e) => {
    const selection = e.target.value;
    setWorkforceSelection(selection);
    if (selection === 'automatic') {
      setSelectedWorkforce(null);
    }
  };

  const handleScheduleSelect = (schedule) => {
    setSelectedSchedule(schedule);
  };

  const handleWorkforceSelectitem = (worker) => {
    handleWorkforceSelect(worker);
  };

  const inputStyle = {
    display: 'block',
    width: '100%',
    padding: '8px',
    marginBottom: '8px',
    borderRadius: '4px',
    border: '1px solid #d9d9d9',
    fontSize: '14px',
    boxSizing: 'border-box'
  };

  const errorText = (msg) => (
    msg ? <div style={{ color: '#e53935', fontSize: 12, marginTop: -4, marginBottom: 8 }}>{msg}</div> : null
  );

  const fieldBorder = (hasErr) => ({
    ...inputStyle,
    borderColor: hasErr ? '#e53935' : '#ced4da',
    borderWidth: hasErr ? '2px' : '1px'
  });

  return (<>
    <div style={{ display: 'flex', height: '100vh', fontFamily: 'sans-serif' }}>
      <Script
        url={`https://maps.googleapis.com/maps/api/js?key=AIzaSyDe-1PlmNj0J4kGAbZBJcelb_mlUof-6us&libraries=places`}
        onLoad={handleScriptLoad}
        onError={handleScriptError}
      />

      {/* Left Sidebar */}
      <div style={{ width: '360px', borderRight: '1px solid #ccc', padding: '1rem', overflowY: 'auto' }}>
        {/* UI banner (no alert dialogs) */}
        {uiMessage && (
          <div
            style={{
              marginBottom: 10,
              padding: '8px 10px',
              borderRadius: 8,
              fontSize: 13,
              background: uiMessage.type === 'success' ? '#e6f4ea' : '#fdecea',
              color: uiMessage.type === 'success' ? '#1e7e34' : '#b71c1c',
              border: `1px solid ${uiMessage.type === 'success' ? '#c7ebd1' : '#f5c6cb'}`
            }}
          >
            {uiMessage.text}
          </div>
        )}

        {/* Tabs (optional) */}
        {!hideSidebarTabs && (
          <div style={{ display: 'flex' }}>
            {[
              // tabs if you want to show
            ].map((item) => (
              <div
                key={item.id}
                onClick={() => setActiveTab(item.id)}
                style={{
                  display: 'flex',
                  alignItems: 'center',
                  padding: '10px 15px',
                  cursor: 'pointer',
                  fontWeight: activeTab === item.id ? 'bold' : 'normal',
                  color: activeTab === item.id ? '#000' : '#777'
                }}
              >
                <span
                  className="material-symbols-outlined"
                  style={{
                    fontSize: '18px',
                    marginRight: '6px',
                    color: activeTab === item.id ? '#000' : '#777'
                  }}
                >
                  {item.icon}
                </span>
                {item.label}
              </div>
            ))}
          </div>
        )}

        {/* Services Search */}
        <div style={{ position: 'relative', marginBottom: '4px' }}>
          <span
            className="material-symbols-outlined"
            style={{
              position: 'absolute',
              left: '10px',
              top: '50%',
              transform: 'translateY(-50%)',
              color: '#666'
            }}
          >
            search
          </span>

          <input
            type="text"
            placeholder="Search services..."
            value={serviceSearchTerm}
            onChange={(e) => setServiceSearchTerm(e.target.value)}
            style={{
              width: '100%',
              padding: '8px 12px 8px 32px',
              borderRadius: '4px',
              border: '1px solid #ddd',
              fontSize: '14px',
              backgroundColor: '#fff'
            }}
          />

          {serviceSearchTerm && (
            <span
              style={{
                position: 'absolute',
                right: '10px',
                top: '50%',
                transform: 'translateY(-50%)',
                cursor: 'pointer',
                color: '#999'
              }}
              onClick={() => setServiceSearchTerm('')}
            >
              ✕
            </span>
          )}
        </div>

        {/* Services Radio Buttons */}
        <div>
          <div style={{ marginBottom: "2px" }}>
            <div
              style={{
                paddingBottom: "10px",
                borderBottom: "1px solid #eee",
                display: "flex",
                alignItems: "center",
              }}
            >
              {servicePage > 0 && (
                <ChevronLeftIcon
                  onClick={handlePrev}
                  style={{
                    cursor: "pointer",
                    marginTop: "10px",
                    marginRight: "8px",
                    color: "#555",
                    fontSize: "20px",
                  }}
                />
              )}

              <div
                style={{
                  display: "flex",
                  flexWrap: "wrap",
                  gap: "1rem",
                  marginTop: "0.5rem",
                  flex: 1,
                }}
              >
                {visibleServices.length > 0 ? (
                  visibleServices.map((service) => (
                    <div
                      key={service.id}
                      style={{
                        width: "calc(33.33% - 0.67rem)",
                        minWidth: "100px"
                      }}
                    >
                      <label
                        style={{
                          display: "flex",
                          alignItems: "center",
                          cursor: "pointer",
                          width: "100%",
                          lineHeight: 1.2
                        }}
                      >
                        <div style={{ display: "flex", alignItems: "center", gap: "2px", flexShrink: 0 }}>
                          <input
                            type="radio"
                            name="service"
                            value={service.id}
                            checked={selectedService?.id === service.id}
                            onChange={() => setSelectedService(service)}
                            style={{
                              accentColor: "#7E57C2",
                              cursor: "pointer"
                            }}
                          />
                          <span
                            className="material-symbols-outlined"
                            style={{
                              color: "#333",
                              fontSize: "22px"
                            }}
                          >
                            {service.icon_name || 'directions_car'}
                          </span>
                        </div>

                        <span
                          title={service.service_title}
                          style={{
                            textTransform: "capitalize",
                            whiteSpace: "nowrap",
                            overflow: "hidden",
                            marginTop: "4px",
                            textOverflow: "ellipsis",
                            flexGrow: 1,
                            flexShrink: 1,
                            minWidth: 0
                          }}
                        >
                          {service.service_title}
                        </span>
                      </label>
                    </div>
                  ))
                ) : (
                  <div style={{ padding: "8px", color: "#999", fontSize: "14px" }}>
                    No services found
                  </div>
                )}
              </div>

              {(servicePage + 1) * servicesPerPage < filteredServices.length && (
                <ChevronRightIcon
                  onClick={handleNext}
                  style={{
                    cursor: "pointer",
                    marginTop: "10px",
                    marginLeft: "8px",
                    color: "#555",
                    fontSize: "20px",
                  }}
                />
              )}
            </div>
          </div>
        </div>

        {/* Sub-Services */}
        {selectedService && (
          <div style={{ marginBottom: '2px' }}>
            <div style={{ marginTop: '2px' }}>
              <div
                style={{
                  paddingBottom: "2px",
                  borderBottom: "1px solid #eee",
                  display: "flex",
                  alignItems: "center",
                }}
              >
                {subServicePage > 0 && (
                  <ChevronLeftIcon
                    onClick={handleSubServicePrev}
                    style={{
                      cursor: "pointer",
                      marginTop: "10px",
                      marginRight: "8px",
                      color: "#555",
                      fontSize: "20px",
                    }}
                  />
                )}

                <div
                  style={{
                    display: "flex",
                    flexWrap: "wrap",
                    gap: "1rem",
                    flex: 1,
                  }}
                >
                  {visibleSubServices.length > 0 ? (
                    visibleSubServices.map((subService) => (
                      <div
                        key={subService.id}
                        style={{
                          width: "calc(33.33% - 0.67rem)",
                          minWidth: "100px"
                        }}
                      >
                        <label
                          style={{
                            display: "flex",
                            alignItems: "center",
                            gap: "2px",
                            cursor: "pointer",
                            width: "100%",
                          }}
                        >
                          <span
                            className="material-symbols-outlined"
                            style={{
                              verticalAlign: "start",
                              marginRight: "6px",
                              color: "#333",
                              fontSize: "24px",
                            }}
                          >
                            {subService.icon_name}
                          </span>

                          <input
                            type="radio"
                            name="subService"
                            value={subService.id}
                            checked={selectedSubService?.id === subService.id}
                            onChange={() => setSelectedSubService(subService)}
                            style={{ accentColor: "#7E57C2" }}
                          />

                          <span style={{ textTransform: "capitalize", whiteSpace: "nowrap", overflow: "hidden", textOverflow: "ellipsis" }}>
                            {subService.service_title}
                          </span>
                        </label>
                      </div>
                    ))
                  ) : (
                    <div style={{ padding: "8px", color: "#999", fontSize: "14px" }}>
                      No sub-services available
                    </div>
                  )}
                </div>

                {(subServicePage + 1) * subServicesPerPage < filteredSubServices.length && (
                  <ChevronRightIcon
                    onClick={handleSubServiceNext}
                    style={{
                      cursor: "pointer",
                      marginTop: "10px",
                      marginLeft: "8px",
                      color: "#555",
                      fontSize: "20px",
                    }}
                  />
                )}
              </div>
            </div>
          </div>
        )}

        {/* Task Section */}
        <div>
          <div style={{
            display: 'flex',
            justifyContent: 'space-between',
            alignItems: 'center',
            marginBottom: '4px',
            fontSize: '16px',
            fontWeight: 'bold',
            color: '#333'
          }}>
            <div>Tasks</div>
            <div style={{ display: 'flex', alignItems: 'center', gap: '16px' }}>
              <button
                onClick={() => setNewTaskFormOpen(!newTaskFormOpen)}
                style={{
                  display: 'flex',
                  alignItems: 'center',
                  backgroundColor: 'transparent',
                  border: 'none',
                  color: '#000',
                  fontSize: '14px',
                  cursor: 'pointer',
                }}
              >
                <span style={{ marginRight: '6px' }}>Add</span>
                <AddIcon style={{ fontSize: '16px', color: '#000' }} />
              </button>
              <button
                onClick={() => setUploadSidebarOpen(true)}
                style={{
                  display: 'flex',
                  alignItems: 'center',
                  backgroundColor: 'transparent',
                  border: 'none',
                  color: '#000',
                  fontSize: '14px',
                  cursor: 'pointer',
                }}
              >
                <span style={{ marginRight: '6px' }}>Upload</span>
                <CloudUploadIcon style={{ fontSize: '16px', color: '#000' }} />
              </button>
            </div>
          </div>

          <div style={{ fontSize: '14px', color: '#6e6e6e', display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
            <div>
              {fileUploaded && (
                <>
                  <span style={{ fontWeight: 600, color: '#333' }}>
                    {tasks.length} Tasks
                  </span>
                  <span> added so far,&nbsp;</span>
                  <a href="#" style={{ color: '#000', textDecoration: 'none' }}>
                    Manage
                  </a>
                  <span className="material-symbols-outlined" style={{
                    verticalAlign: 'middle',
                    marginLeft: '8px',
                    color: '#333',
                    fontSize: '18px',
                    cursor: 'pointer'
                  }}>
                    edit
                  </span>
                  <span className="material-symbols-outlined" style={{
                    verticalAlign: 'middle',
                    marginLeft: '4px',
                    color: '#333',
                    fontSize: '18px',
                    cursor: 'pointer'
                  }}>
                    delete
                  </span>
                </>
              )}
            </div>
          </div>

          {/* New Task Form (inline fields, no Save button) */}
          {newTaskFormOpen && (
            <div style={{ borderBottom: '1px solid #ccc'}}>
              <div style={{ fontWeight: 'bold', marginTop: '10px' }}>New Task</div>

              {/* Customer Search */}
              <div style={{ marginTop: '10px', position: 'relative' }}>
                <input
                  name="customer"
                  value={taskFormData.customer}
                  onChange={handleCustomerSearch}
                  onFocus={() => {
                    if (taskFormData.customer.length > 1 && customerOptions.length > 0) {
                      setShowCustomerDropdown(true);
                    }
                  }}
                  onBlur={() => setTimeout(() => setShowCustomerDropdown(false), 200)}
                  style={fieldBorder(!!formErrors.customer)}
                  placeholder="Search customer"
                  ref={customerInputRef}
                  autoComplete="off"
                />
                {errorText(formErrors.customer)}
                {showCustomerDropdown && customerOptions.length > 0 && (
                  <div style={{
                    position: 'absolute',
                    zIndex: 1000,
                    width: '100%',
                    maxHeight: '200px',
                    overflowY: 'auto',
                    backgroundColor: 'white',
                    border: '1px solid #d9d9d9',
                    borderRadius: '4px',
                    boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
                    marginTop: '2px'
                  }}>
                    {customerOptions.map(customer => (
                      <div
                        key={customer.id}
                        style={{
                          padding: '8px 12px',
                          cursor: 'pointer',
                          borderBottom: '1px solid #f8f9fa',
                        }}
                        onMouseDown={(e) => {
                          e.preventDefault();
                          handleCustomerSelect(customer);
                        }}
                      >
                        {customer.first_name} {customer.last_name} ({customer.email})
                      </div>
                    ))}
                  </div>
                )}
              </div>

              {/* Pickup Location */}
              <div style={{ marginBottom: '10px', position: 'relative' }}>
                <input
                  name="location1"
                  value={taskFormData.location1}
                  onChange={handleInputChange}
                  style={{
                    ...fieldBorder(!!formErrors.location1),
                    padding: '8px 12px',
                    height: '38px'
                  }}
                  placeholder={!googleMapsReady ? "Loading maps..." : "Search pickup location"}
                  ref={pickupInputRef}
                  autoComplete="off"
                  id="pickup-location-input"
                  disabled={!googleMapsReady}
                />
                {errorText(formErrors.location1)}
                {!googleMapsReady && (
                  <span style={{
                    position: 'absolute',
                    right: '10px',
                    top: '50%',
                    transform: 'translateY(-50%)',
                    color: '#ff4d4f',
                    fontSize: '12px'
                  }}>
                    Loading maps...
                  </span>
                )}
              </div>

              {/* Drop Location */}
              <div style={{ marginBottom: '10px', position: 'relative' }}>
                <input
                  name="location2"
                  value={taskFormData.location2}
                  onChange={handleInputChange}
                  style={{
                    ...fieldBorder(!!formErrors.location2),
                    padding: '8px 12px',
                    height: '38px'
                  }}
                  placeholder={!googleMapsReady ? "Loading maps..." : "Search drop location"}
                  ref={dropInputRef}
                  autoComplete="off"
                  id="drop-location-input"
                  disabled={!googleMapsReady}
                />
                {errorText(formErrors.location2)}
                {!googleMapsReady && (
                  <span style={{
                    position: 'absolute',
                    right: '10px',
                    top: '50%',
                    transform: 'translateY(-50%)',
                    color: '#ff4d4f',
                    fontSize: '12px'
                  }}>
                    Loading maps...
                  </span>
                )}
              </div>

              {/* Date Time Picker */}
              <div style={{ marginBottom: '10px' }}>
                <input
                  type="datetime-local"
                  name="dateTime"
                  value={taskFormData.dateTime}
                  onChange={handleInputChange}
                  style={fieldBorder(!!formErrors.dateTime)}
                />
                {errorText(formErrors.dateTime)}
              </div>

              {/* No Save button here. Validation happens on Go. */}
            </div>
          )}

          {/* Schedule Settings */}
          <div style={{ marginTop: '1rem' }}>
            <b>Schedule Settings</b>
            <div style={{ marginTop: '0.5rem', accentColor: "#7E57C2" }}>
              <div>
                <label>Schedule Type:</label><br />
                <input
                  type="radio"
                  name="type"
                  checked={scheduleType === 'new'}
                  onChange={() => handleScheduleTypeChange('new')}
                /> New Schedule{' '}
                <input
                  type="radio"
                  name="type"
                  checked={scheduleType === 'existing'}
                  onChange={() => handleScheduleTypeChange('existing')}
                /> Existing Schedule
              </div>

              <div style={{ marginTop: '0.5rem', accentColor: "#7E57C2" }}>
                <label>Allocation:</label><br />
                <input
                  type="radio"
                  name="alloc"
                  value="individual"
                  checked={allocationType === 'individual'}
                  onChange={() => setAllocationType('individual')}
                /> Individual{' '}
                <input
                  type="radio"
                  name="alloc"
                  value="distribute"
                  checked={allocationType === 'distribute'}
                  onChange={() => setAllocationType('distribute')}
                /> Distribute
              </div>
              <div style={{ marginTop: '0.5rem', accentColor: "#7E57C2" }}>
                <label>Workforce Selection:</label><br />
                <input
                  type="radio"
                  name="worker"
                  value="automatic"
                  checked={workforceSelection === 'automatic'}
                  onChange={handleWorkforceSelectionChange}
                /> Automatic{' '}
                <input
                  type="radio"
                  name="worker"
                  value="manual"
                  checked={workforceSelection === 'manual'}
                  onChange={handleWorkforceSelectionChange}
                /> Manual
              </div>
            </div>
          </div>

          <button
            onClick={handleGoClick}
            style={{
              padding: '8px 6px',
              backgroundColor: '#E0E0E0',
              border: 'none',
              color: '#000',
              borderRadius: '6px',
              fontWeight: 'bold',
              cursor: 'pointer',
              boxShadow: '0 2px 4px rgba(0,0,0,0.15)',
              float: 'right',
              marginBottom: '8px',
              position: 'relative',
              top: '-2px'
            }}
          >
            Go
          </button>

          {activeTab === 'dispatch' && (
            <div style={{ marginTop: '1rem', padding: '12px', border: '1px solid #eee', borderRadius: '8px' }}>
              <div style={{ fontWeight: 600, marginBottom: 8 }}>Dispatch</div>
              <div style={{ fontSize: 13, color: '#555', marginBottom: 8 }}>
                <div>Schedule: <b>{(selectedSchedule?.schedule_name) || (createdScheduleId ? `New #${createdScheduleId}` : '—')}</b></div>
                <div>Workforce: <b>{selectedWorkforce ? (selectedWorkforce.first_name || selectedWorkforce.name || selectedWorkforce.id) : (workforceSelection === 'automatic' ? 'Automatic' : '—')}</b></div>
              </div>
              <button
                onClick={handleDispatch}
                disabled={isDispatching}
                style={{ padding: '10px 12px', borderRadius: 8, border: 'none', background: '#34c759', color: 'white', fontWeight: 600, cursor: 'pointer' }}
              >
                {isDispatching ? 'Dispatching…' : 'Dispatch Now'}
              </button>
            </div>
          )}

        </div>
      </div>

      {/* Combined Sidebar */}
      {showCombinedSidebar && (
  <div
    style={{
      width: '360px',
      minWidth: '360px',
      maxWidth: '360px',
      height: '100vh',
      display: 'flex',
      flexDirection: 'column',
      background: '#fff',
      position: 'relative',
      borderRight: '1px solid #ccc',
      overflow: 'hidden',
    }}
  >
    {/* Header */}
    <div
      style={{
        display: 'flex',
        justifyContent: 'space-between',
        alignItems: 'center',
        padding: '0.5rem 1rem',
        background: '#fff',
        zIndex: 2,
      }}
    >
      <h3 style={{ margin: 0, fontSize: '14px', fontWeight: '600' }}>
        Scheduling
      </h3>
      <button
        onClick={() => setShowCombinedSidebar(false)}
        style={{
          background: 'none',
          border: 'none',
          cursor: 'pointer',
          fontSize: '16px',
          color: '#444',
          lineHeight: 1,
          padding: '4px',
        }}
      >
        ×
      </button>
    </div>

    {/* Content Area */}
    <div
      style={{
        flex: 1,
        overflowY: 'scroll',     // always show scrollbar
        padding: '0 1rem',
        paddingBottom: '84px',   // prevent overlap with dispatch bar
      }}
    >
      {/* Workforce Section */}
      {workforceSelection === 'manual' && (
        <div style={{ marginBottom: '1rem' }}>
          <div style={{ position: 'relative', marginBottom: '4px' }}>
            <span
              className="material-symbols-outlined"
              style={{
                position: 'absolute',
                left: '10px',
                top: '50%',
                transform: 'translateY(-50%)',
                color: '#666',
              }}
            >
              search
            </span>

            <input
              type="text"
              placeholder="Search for Team & Services"
              value={serviceSearchTerm}
              onChange={(e) => setServiceSearchTerm(e.target.value)}
              style={{
                width: '100%',
                padding: '8px 12px 8px 32px',
                borderRadius: '4px',
                border: '1px solid #ddd',
                fontSize: '14px',
                backgroundColor: '#fff',
              }}
            />

            {serviceSearchTerm && (
              <span
                style={{
                  position: 'absolute',
                  right: '10px',
                  top: '50%',
                  transform: 'translateY(-50%)',
                  cursor: 'pointer',
                  color: '#999',
                }}
                onClick={() => setServiceSearchTerm('')}
              >
                ✕
              </span>
            )}
          </div>

          {loadingWorkforce ? (
            <div style={{ padding: '10px', textAlign: 'center' }}>
              Loading workforce...
            </div>
          ) : workforceData.length > 0 ? (
            <div style={{ maxHeight: '200px', overflowY: 'scroll' }}>
              {workforceData.map((worker) => (
                <div
                  key={worker.id}
                  style={{
                    padding: '10px 0',
                    marginBottom: '4px',
                    display: 'flex',
                    alignItems: 'center',
                  }}
                >
                  <input
                    type="checkbox"
                    checked={selectedWorkforce?.id === worker.id}
                    onChange={() => setSelectedWorkforce(worker)}
                    style={{ marginRight: '10px', accentColor: '#7E57C2' }}
                  />
                  <div style={{ display: 'flex', alignItems: 'center', width: '100%' }}>
                    <div
                      style={{
                        width: '32px',
                        height: '32px',
                        borderRadius: '50%',
                        backgroundColor: getRandomColor(),
                        color: 'black',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        marginRight: '10px',
                        fontWeight: 'bold',
                      }}
                    >
                      {worker.first_name ? worker.first_name.charAt(0).toUpperCase() : '?'}
                    </div>
                    <div>
                      <div
                        style={{
                          fontWeight: 'bold',
                          cursor: 'pointer',
                          color: hoveredWorkerId === worker.id ? '#1976d2' : '#000',
                        }}
                        onMouseEnter={() => setHoveredWorkerId(worker.id)}
                        onMouseLeave={() => setHoveredWorkerId(null)}
                        onClick={() => handleWorkforceSelect(worker)}
                      >
                        {worker.first_name || 'Unknown'}
                      </div>

                      <div style={{ fontSize: '12px', color: '#666' }}>
                        Schedules: {worker.total_schedules || 0}
                        <span style={{ textDecoration: 'underline' }}> Tasks: {worker.total_tasks || 0}</span>,
                        <span
                          style={{
                            marginLeft: '6px',
                            cursor: 'pointer',
                            color: hoveredWorkerId === `settings-${worker.id}` ? '#1976d2' : '#666',
                            fontWeight: '500',
                            textDecoration: 'underline',
                          }}
                          onMouseEnter={() => setHoveredWorkerId(`settings-${worker.id}`)}
                          onMouseLeave={() => setHoveredWorkerId(null)}
                          onClick={() => {
                            setRightSidebarOpen(true);
                            fetchWorkforceDetails(worker.id);
                          }}
                        >
                          Settings
                        </span>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div style={{ padding: '10px', textAlign: 'center', color: '#666' }}>
              No workforce available
            </div>
          )}
        </div>
      )}

      {/* Workforce Details Section */}
      {selectedWorkforce && (
        <div style={{ marginTop: '1rem', borderTop: '1px solid #eee' }}>
          <h4 style={{ marginBottom: '0.5rem' }}>Workforce Details</h4>

          {loadingWorkforceDetails ? (
            <div style={{ padding: '10px', textAlign: 'center' }}>Loading details...</div>
          ) : selectedWorkforceDetails ? (
            <>
              {/* Basic Info */}
              <div style={{ display: 'flex', alignItems: 'center', marginBottom: '16px' }}>
                <div
                  style={{
                    width: '40px',
                    height: '40px',
                    borderRadius: '50%',
                    backgroundColor: '#E0E0E0',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    marginRight: '10px',
                    fontWeight: 'bold',
                  }}
                >
                  {selectedWorkforce.first_name?.charAt(0).toUpperCase()}
                </div>
                <div>
                  <div style={{ fontWeight: 'bold' }}>
                    {selectedWorkforce.first_name} {selectedWorkforce.last_name}
                  </div>
                </div>
              </div>

              {/* Tasks */}
              {(() => {
                const demoTasks = [
                  {
                    customer_info: { name: 'Customer ABCDE', location: 'Location 1' },
                    scheduled_start: '2025-08-19T11:00:00',
                    status: 'in_progress',
                    fleet_info: { number: 'Truck 0001' },
                  },
                  {
                    customer_info: { name: 'Customer XYZ', location: 'Location 2' },
                    scheduled_start: '2025-08-20T14:30:00',
                    status: 'pending',
                    fleet_info: { number: 'Truck 0002' },
                  },
                ];

                const tasksToShow =
                  selectedWorkforceDetails.tasks?.length > 0
                    ? selectedWorkforceDetails.tasks
                    : demoTasks;

                return (
                  <>
                    <h5 style={{ margin: '12px 0 8px 0', fontWeight: 600 }}>Tasks</h5>
                    {tasksToShow.map((task, i) => {
                      const fleet =
                        task.fleet_info && Object.keys(task.fleet_info).length > 0
                          ? task.fleet_info
                          : { number: 'Truck 0001' };

                      return (
                        <div
                          key={i}
                          style={{
                            display: 'flex',
                            justifyContent: 'space-between',
                            alignItems: 'center',
                            padding: '8px 0',
                            borderBottom: '1px solid #eee',
                          }}
                        >
                          {/* Left */}
                          <div style={{ display: 'flex', alignItems: 'center', flex: 1 }}>
                            <div
                              style={{
                                width: '16px',
                                height: '16px',
                                borderRadius: '4px',
                                background: '#E0E0E0',
                                marginRight: '10px',
                              }}
                            />
                            <div>
                              <div style={{ fontWeight: 'bold' }}>
                                {task.customer_info?.name || 'Customer N/A'}
                              </div>
                              <div style={{ fontSize: '12px', color: '#666' }}>
                                {task.customer_info?.location || 'Coimbatore International Airport, Avinashi Road.'}
                              </div>
                              <div style={{ fontSize: '12px', color: '#666' }}>
                                {task.scheduled_start
                                  ? new Date(task.scheduled_start).toLocaleString()
                                  : '24/07/2025-12:32PM'}
                              </div>
                            </div>
                          </div>

                          {/* Right */}
                          <div style={{ display: 'flex', alignItems: 'center' }}>
                            <div
                              style={{
                                width: '28px',
                                height: '28px',
                                borderRadius: '50%',
                                backgroundColor: '#E0E0E0',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                marginRight: '8px',
                                fontWeight: 'bold',
                                fontSize: '12px',
                              }}
                            >
                              {fleet.number ? fleet.number.charAt(0).toUpperCase() : 'T'}
                            </div>
                            <span style={{ fontSize: '13px', fontWeight: 500 }}>
                              {fleet.number || 'Truck N/A'}
                            </span>
                          </div>
                        </div>
                      );
                    })}
                  </>
                );
              })()}
            </>
          ) : (
            <div style={{ padding: '10px', textAlign: 'center', color: '#666' }}>
              No details available
            </div>
          )}
        </div>
      )}
    </div>

    {/* Dispatch Button */}
    <div
      style={{
        position: 'absolute',
        bottom: 0,
        left: 0,
        width: '100%',
        backgroundColor: '#fff',
        padding: '12px 16px',
        display: 'flex',
        justifyContent: 'flex-end',
      }}
    >
      <button
        onClick={handleDispatch}
        disabled={isDispatching}
        style={{
          padding: '10px 24px',
          backgroundColor: '#E0E0E0',
          border: 'none',
          borderRadius: '6px',
          color: '#000',
          cursor: isDispatching ? 'not-allowed' : 'pointer',
          width: 'auto',
          minWidth: '100px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.15)',
          marginRight: '30px',
        }}
      >
        {isDispatching ? 'Dispatching...' : 'Dispatch'}
      </button>
    </div>

    {/* Custom Scrollbar */}
    <style>{`
      ::-webkit-scrollbar { width: 6px; }
      ::-webkit-scrollbar-thumb { background: #bbb; border-radius: 3px; }
      ::-webkit-scrollbar-thumb:hover { background: #999; }
    `}</style>
  </div>
)}



      {/* Middle Scheduling Area with Google Map */}
      <div style={{
        flex: 1,
        position: 'relative',
        overflow: 'hidden'
      }}>
        <div
          onClick={() => setRightSidebarOpen(!rightSidebarOpen)}
          style={{
            position: 'absolute',
            top: '20px',
            right: rightSidebarOpen ? '35px' : '10px',
            zIndex: 1000,
            background: 'white',
            borderRadius: '50%',
            width: '40px',
            height: '40px',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            boxShadow: '0 2px 5px rgba(0,0,0,0.2)',
            cursor: 'pointer',
            transition: 'right 0.3s ease'
          }}
        >
          {rightSidebarOpen ? '❯' : '❮'}
        </div>

        <GoogleMapReact
          bootstrapURLKeys={{ key: 'AIzaSyDe-1PlmNj0J4kGAbZBJcelb_mlUof-6us', libraries: ['places']  }}
          yesIWantToUseGoogleMapApiInternals
          onGoogleApiLoaded={({ map, maps }) => {
            setGoogleMapsReady(true);
            setMapsLoading(false);
          }}
          center={mapCenter}
          zoom={mapZoom}
          onClick={handleMapClick}
          options={{
            fullscreenControl: false,
            streetViewControl: false,
            mapTypeControl: false,
            zoomControl: false,
            styles: [
              { featureType: "poi", elementType: "labels", stylers: [{ visibility: "off" }] }
            ]
          }}
          style={{ height: '100%', width: '100%' }}
        />
      </div>

      {/* Right Sidebar */}
     <div
  style={{
    width: rightSidebarOpen ? "350px" : "0",
    transition: "width 0.3s ease",
    background: "#fff",
    borderLeft: "1px solid #ccc",
    overflow: "hidden",
  }}
>
  {rightSidebarOpen && (
    <div
      style={{
        width: "360px",
        minWidth: "360px",
        maxWidth: "360px",
        height: "100vh",
        display: "flex",
        flexDirection: "column",
        background: "#fff",
        position: "relative",
        borderRight: "1px solid #ccc",
        overflow: "hidden",
        
      }}
    >
      {/* Header */}
      <div
        style={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
        }}
      >
      </div>

      {/* ---------- TOP HALF (Schedules / Tasks Tabs) ---------- */}
      <div
        style={{
          flex: 1,
          overflowY: "auto",
          padding: "1rem",
          paddingTop: "12px",
        }}
      >
        {loadingWorkforceDetails ? (
          <div style={{ padding: "10px", textAlign: "center" }}>
            Loading details...
          </div>
        ) : selectedWorkforceDetails ? (
          <>
            {/* Tabs (screenshot style) */}
            <div
              style={{
                display: "flex",
                paddingBottom: "6px",
              }}
            >
              {[
                { key: "schedules", label: "Schedule" },
                { key: "tasks", label: "Tasks" },
              ].map((tab) => {
                const isActive = activeTab === tab.key;
                return (
                  <button
                    key={tab.key}
                    onClick={() => setActiveTab(tab.key)}
                    style={{
                      appearance: "none",
                      background: "transparent",
                      border: "none",
                      padding: 0,
                      margin: 0,
                      cursor: "pointer",
                      fontSize: "14px",
                      fontWeight: isActive ? 600 : 500,
                      color: "#000",
                      position: "relative",
                    }}
                  >
                    {tab.label}
                    {isActive && (
                      <div
                        style={{
                          height: "2px",
                          background: "#000",
                          marginTop: "3px",
                          borderRadius: "1px",
                        }}
                      />
                    )}
                  </button>
                );
              })}
            </div>

            {/* Tab Content */}
            {activeTab === "schedules" && (
              <div>
                {selectedWorkforceDetails.schedules?.map((s) => (
                  <div
                    key={s.schedule_id}
                    style={{
                      display: "flex",
                      alignItems: "center",
                      padding: "8px 0",
                      borderBottom: "1px solid #eee",
                    }}
                  >
                    <div
                      style={{
                        width: "32px",
                        height: "32px",
                        borderRadius: "50%",
                        backgroundColor: "#E0E0E0",
                        color: "#000",
                        display: "flex",
                        alignItems: "center",
                        justifyContent: "center",
                        fontWeight: "bold",
                        marginRight: "10px",
                      }}
                    >
                      {s.schedule_name?.charAt(0).toUpperCase() || "S"}
                    </div>
                    <div>
                      <div style={{ fontWeight: "bold" }}>
                        {s.schedule_name}
                      </div>
                      <div style={{ fontSize: "12px", color: "#666" }}>
                        {s.start_time
                          ? new Date(s.start_time).toLocaleString()
                          : "No start time"}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}

            {activeTab === "tasks" && (
              <div>
                {selectedWorkforceDetails.tasks?.map((t) => (
                  <div
                    key={t.task_id}
                    style={{
                      display: "flex",
                      alignItems: "center",
                      padding: "8px 0",
                      borderBottom: "1px solid #eee",
                    }}
                  >
                    <div
                      style={{
                        width: "32px",
                        height: "32px",
                        borderRadius: "50%",
                        backgroundColor: "#E0E0E0",
                        color: "#000",
                        display: "flex",
                        alignItems: "center",
                        justifyContent: "center",
                        fontWeight: "bold",
                        marginRight: "10px",
                      }}
                    >
                      {t.customer_info?.first_name?.charAt(0).toUpperCase() ||
                        "T"}
                    </div>
                    <div>
                      <div style={{ fontWeight: "bold" }}>
                        {t.customer_info?.first_name}{" "}
                        {t.customer_info?.last_name}
                      </div>
                      <div style={{ fontSize: "12px", color: "#666", display: "flex", alignItems: "center", gap: "4px" }}>
  <span className="material-symbols-outlined" style={{ fontSize: "14px", color: "#666" }}>
    call
  </span>
  {t.customer_info?.phone || "N/A"}
</div>

<div style={{ fontSize: "12px", color: "#666", display: "flex", alignItems: "center", gap: "4px" }}>
  <span className="material-symbols-outlined" style={{ fontSize: "14px", color: "#666" }}>
    mail
  </span>
  {t.customer_info?.email || "N/A"}
</div>

                    </div>
                  </div>
                ))}
              </div>
            )}
          </>
        ) : (
          <div
            style={{ padding: "10px", textAlign: "center", color: "#666" }}
          >
            No details available
          </div>
        )}
      </div>

      {/* ---------- BOTTOM HALF (Google Calendar) ---------- */}
      <div
  style={{
    flex: "0 0 320px",            // reservér 520px høyde til kalenderen
    borderTop: "1px solid #eee",
    padding: "10px",
    display: "flex",
    flexDirection: "column",
    minHeight: 0
  }}
>
  <h4 style={{ marginBottom: "8px", display: "flex", alignItems: "center", gap: "6px" }}>
    <span className="material-symbols-outlined" style={{ fontSize: "18px", color: "#000" }}>
      calendar_month
    </span>
    Calendar
  </h4>

  <div style={{ flex: 1, minHeight: 0 }}>
    <iframe
      src="https://calendar.google.com/calendar/embed?src=en.indian%23holiday%40group.v.calendar.google.com&ctz=Asia%2FKolkata"
      style={{
        border: "0",
        width: "100%",
        height: "100%",          // fyll containeren
        borderRadius: "6px",
      }}
      frameBorder="0"
      scrolling="auto"            // tillat rulling inne i iframe
      title="Google Calendar"
    />
  </div>
</div>

    </div>
  )}
</div>



      {/* Upload Sidebar */}
     {uploadSidebarOpen && (
  <div
    style={{
      width: "360px",
      minWidth: "360px",
      maxWidth: "360px",
      height: "100vh",
      display: "flex",
      flexDirection: "column",
      background: "#fff",
      position: "relative",
      borderRight: "1px solid #ccc",
      overflow: "hidden",
      padding: "1rem",
    }}
  >
    {/* Header */}
   <div
  style={{
    display: "flex",
    alignItems: "center",
    justifyContent: "space-between",
    marginBottom: "20px",
  }}
>
  <h4 style={{ margin: 0 }}>Upload Tasks</h4>

  <button
    onClick={() => setUploadSidebarOpen(false)}
    style={{
      background: "none",
      border: "none",
      fontSize: "16px",
      cursor: "pointer",
      lineHeight: 1,
      color: "#000",     // black ×
      marginRight: 0     // no extra margin
    }}
    aria-label="Close upload sidebar"
  >
    ×
  </button>
</div>


    {/* Upload Section */}
    <div style={{ marginBottom: "20px" }}>
    
      <div
        style={{
          border: "2px dashed #ccc",
          padding: "20px",
          textAlign: "center",
          borderRadius: "5px",
        }}
      >
        <CloudUploadIcon style={{ fontSize: "48px", marginBottom: "10px" }} />
        <p style={{ margin: "0 0 10px" }}>
          Drag and drop your file here or
        </p>

        <label
          htmlFor="file-upload-sidebar"
          style={{
            display: "inline-block",
            padding: "6px 16px",
            backgroundColor: uploading ? "#999" : "#E0E0E0",
            color: "black",
            borderRadius: "4px",
            cursor: uploading ? "not-allowed" : "pointer",
            opacity: uploading ? 0.7 : 1,
          }}
        >
          {uploading ? "Uploading…" : "Browse Files"}
        </label>

        <input
          id="file-upload-sidebar"
          type="file"
          accept=".xls,.xlsx"
          style={{ display: "none" }}
          onChange={handleFileUpload}
          disabled={uploading}
        />
      </div>
    </div>

    {/* Instructions */}
    <div style={{ marginBottom: "20px" }}>
      <h4 style={{ margin: "0 0 10px" }}>Upload Instructions</h4>
      <ul style={{ paddingLeft: "20px", margin: 0 }}>
        <li style={{ marginBottom: "8px" }}>.xls or .xlsx only</li>
        <li style={{ marginBottom: "8px" }}>Max 5MB</li>
        <li style={{ marginBottom: "8px" }}>Fill all required fields</li>
        <li>Follow the sample format</li>
      </ul>
    </div>

    {/* Download Sample */}
    <div>
      <h4 style={{ margin: "0 0 10px" }}>Download Sample File</h4>

      {/* Option A: static file from /public/sample.xls */}
      <a
        href="/sample.xlsx"
        download="Sample.xlsx"
        style={{
          display: "inline-flex",
          alignItems: "center",
          padding: "8px 16px",
          backgroundColor: "#E0E0E0",
          color: "black",
          border: "none",
          borderRadius: "4px",
          cursor: "pointer",
          textDecoration: "none",
          font: "inherit",
        }}
      >
        <CloudUploadIcon style={{ marginRight: "8px" }} />
        Download Sample.xls
      </a>

      {/* If you prefer a button, use onClick with programmatic download:
      <button
        onClick={() => {
          const a = document.createElement("a");
          a.href = "/sample.xls";
          a.download = "Sample.xls";
          document.body.appendChild(a);
          a.click();
          a.remove();
        }}
        style={{
          display: "flex",
          alignItems: "center",
          padding: "8px 16px",
          backgroundColor: "#34c759",
          color: "white",
          border: "none",
          borderRadius: "4px",
          cursor: "pointer",
          marginTop: "10px"
        }}
      >
        <CloudUploadIcon style={{ marginRight: "8px" }} />
        Download Sample.xls
      </button>
      */}
    </div>
  </div>
)}



    </div>
  </>);
};

const getRandomColor = () => {
  const colors = [
    '#E0E0E0'
  ];
  return colors[Math.floor(Math.random() * colors.length)];
};

export default DispatchPage;
