// src/components/pages/Dashboard/index.js
import React, { useState, useEffect, useRef } from 'react';
import '../../themes/blue/Dashboard.css';
import axios from 'axios';
import Chart from 'chart.js/auto';
import queryString from 'query-string';
import { useLocation } from 'react-router-dom';
//import DatePicker from 'react-datepicker';


const Dashboard = ({ projectId = '' }) => { 
    const accountId = localStorage.getItem('account_id') || null;
    const onboardingKey = accountId + '_setup_done';
    const [setupDone, setSetupDone] = useState(() => localStorage.getItem(onboardingKey) || '0');
    const name = localStorage.getItem('name') || 'User';
    const [dashboardData, setDashboardData] = useState({
        active_projects: 0,
        total_interactions: 0,
        total_goals: 0,
        total_visitors: 0,
        total_visits: 0,
        total_system_messages: 0, // Add this line
        graph_data: [],
        daily_responses: [],
    });
    const chartRef = useRef(null);
    const barChartRef = useRef(null);
    const chartInstanceRef = useRef(null); // Ref to store the chart instance
    const barChartInstanceRef = useRef(null); // Ref to store the chart instance
    const location = useLocation();
    const queryParams = queryString.parse(location.search);
    const message = queryParams.message || null;

    //console.log("project_ id ", projectId);
    // Calculate dates for the past month
    const oneMonthAgo = new Date();
    oneMonthAgo.setMonth(oneMonthAgo.getMonth() - 1);
    const currentDate = new Date();

    // Format dates as YYYY-MM-DD
    const formatDate = (date) => {
        //console.log(date);
        return date.toISOString().split('T')[0];
    };
    const formatDateNew = (date) => {
        const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
        const monthName = months[date.getMonth()];
        const day = date.getDate();
    
        return `${monthName} ${day}`;
    };

    const [dateFrom, setDateFrom] = useState(formatDate(oneMonthAgo));
    const [dateTo, setDateTo] = useState(formatDate(currentDate));
    const dateFromRef = useRef(null);
    const dateToRef = useRef(null);

    const handleDateFromClick = () => {
        dateFromRef.current.focus();
    };

    const handleDateToClick = () => {
        dateToRef.current.focus();
    };
    const [activeAgents, setActiveAgents] = useState(parseInt(localStorage.getItem('active_agents') || 0));
    const showInitialSetup = !projectId;
    const showWelcomeSection = !projectId &&  (activeAgents === 0 && setupDone === '0');
    
    // Fetching dashboard data from the API
    useEffect(() => {
        const fetchData = async () => {
            const cacheKey = `dashboardData_${accountId}_${dateFrom}_${dateTo}_${projectId || 'global'}`;
            const cachedData = sessionStorage.getItem(cacheKey);
            const currentTime = new Date().getTime();
            const cacheTimeKey = `${cacheKey}_time`;
            const cachedTime = sessionStorage.getItem(cacheTimeKey);
        
            if (cachedData && cachedTime && currentTime - parseInt(cachedTime, 10) < 3600000) { // 1 hour cache validity
                //console.log('Using cached data for the specific date range and project');
                setDashboardData(JSON.parse(cachedData));
            } else {
                try {
                    const response = await axios.get(`/dashboard/${accountId}/?from=${dateFrom}&to=${dateTo}${projectId ? `&project_id=${projectId}` : ''}`);
                    //console.log('Fetching new data');
                    setDashboardData(response.data);
                    // Cache the new data with a unique key based on accountId, date range, and projectId
                    sessionStorage.setItem(cacheKey, JSON.stringify(response.data));
                    sessionStorage.setItem(cacheTimeKey, currentTime.toString());
                } catch (error) {
                    console.error('Error fetching dashboard data:', error);
                }
            }
        };
                    
        fetchData();
        //console.log("dashdata ", dashboardData);
    }, [accountId, dateFrom, dateTo, projectId]);
    
    useEffect(() => {
            
        if (dashboardData.graph_data) {
            const chartCtx = chartRef.current.getContext('2d');
            if (chartInstanceRef.current) {
                chartInstanceRef.current.destroy();
            }
    
            chartInstanceRef.current = new Chart(chartCtx, {
                type: 'line',
                data: {
                    labels: dashboardData.graph_data.map(data => formatDateNew(new Date(data.date))),
                    datasets: [
                        {
                            label: 'Interactions',
                            data: dashboardData.graph_data.map(data => data.interactions),
                            borderColor: 'rgb(75, 192, 192)',
                            backgroundColor: 'rgba(75, 192, 192, 0.5)',
                            fill: true
                        },
                        {
                            label: 'Visitors',
                            data: dashboardData.graph_data.map(data => data.visitors),
                            borderColor: 'rgb(255, 99, 132)',
                            backgroundColor: 'rgba(255, 99, 132, 0.5)',
                            fill: true
                        }
                    ]
                },
                options: {
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                display: false // Remove y-axis background lines
                            }
                        },
                        x: {
                            grid: {
                                display: false // Remove x-axis background lines
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: true,
                            position: 'bottom'
                        },
                        title: {
                            display: true,
                            text: 'Visitors Vs Interactions'
                        },
                        tooltip: {
                            callbacks: {
                                label: (context) => {
                                    let label = context.dataset.label || '';
                                    if (label) {
                                        label += ': ';
                                    }
                                    if (context.parsed.y !== null) {
                                        label += context.parsed.y.toLocaleString();
                                    }
                                    return label;
                                },
                                title: (tooltipItem) => formatDateNew(new Date(tooltipItem[0].label))
                            }
                        }
                    }
                }
            });
        }
    
        return () => {
            if (chartInstanceRef.current) {
                chartInstanceRef.current.destroy();
            }
        };
    
    }, [dashboardData.graph_data]);
    
    useEffect(() => {

        //console.log("Rendering chart with data:", dashboardData.daily_responses);

        if (dashboardData.daily_responses && dashboardData.daily_responses.length > 0) {
            const chartCtx = barChartRef.current.getContext('2d');
            if (barChartInstanceRef.current) {
                barChartInstanceRef.current.destroy();
            }
    
            const labels = dashboardData.daily_responses.map(response => formatDateNew(new Date(response.date)));
            const dataPoints = dashboardData.daily_responses.map(response => response.responses);
    
            barChartInstanceRef.current = new Chart(chartCtx, {
                type: 'bar',
                data: {
                    labels: labels, // Dates as labels for each bar
                    datasets: [{
                        label: 'Daily Responses',
                        data: dataPoints, // Number of system messages for each day
                        backgroundColor: 'rgba(54, 162, 235, 0.5)',
                        borderColor: 'rgb(54, 162, 235)',
                        borderWidth: 0
                    }]
                },
                options: {
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                display: false // Remove y-axis background lines
                            }
                        },
                        x: {
                            grid: {
                                display: false // Remove x-axis background lines
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: true,
                            position: 'bottom'
                        },
                        title: {
                            display: true,
                            text: 'Daily Responses'
                        },
                        tooltip: {
                            callbacks: {
                                label: (context) => {
                                    let label = context.dataset.label || '';
                                    if (label) {
                                        label += ': ';
                                    }
                                    if (context.parsed.y !== null) {
                                        label += context.parsed.y.toLocaleString();
                                    }
                                    return label;
                                }
                            }
                        }
                    }
                }
            });
        }
    
        return () => {
            if (barChartInstanceRef.current) {
                barChartInstanceRef.current.destroy();
            }
        };
    }, [dashboardData.daily_responses]); // Depend on daily_responses for re-render
    

    const hideOnboarding = () => {
        localStorage.setItem(onboardingKey, '1');
        setSetupDone('1');
    };

    return (
        <div className="dashboard-container">
            {!projectId && showInitialSetup && (
            <h1>{name}, Welcome Back!</h1>
            )}
            {message && (
              <p className='success'> {message}</p>
            )}

            { showWelcomeSection && (
                <div className="welcome-section">
                <strong>Welcome to the thought platform</strong>
                <p>Congratulations! Your account is now setup to start creating your CX platform. Embark on your journey by creating your very first CX AI Agent. </p>
                <div className="welcome-cards-container">
                    <div className="welcome-card">
                        <h3><span className="material-symbols-outlined">support_agent</span>Agents</h3>
                        <p>Craft your AI agents with precision, infusing them with your unique content and design.</p>
                        <a href="/agents">Create an Agent</a>
                    </div>

                    <div className="welcome-card">
                        <h3><span className="material-symbols-outlined">air</span>Flows</h3>
                        <p>Effortlessly orchestrate and control your workflows to achieve a seamless, unified experience. </p>
                        <a href="/flows">New Flow</a>
                    </div>

                    <div className="welcome-card">
                        <h3><span className="material-symbols-outlined">apps</span>Apps & Integrations</h3>
                        <p>Harmoniously integrate with a suite of applications, ensuring a cohesive and streamlined experience throughout.</p>
                        <a href="/apps">Integrate</a>
                    </div>

                    <div className="welcome-card">
                        <h3><span className="material-symbols-outlined">group</span>Team</h3>
                        <p>Invite, collaborate, and thrive with your team members. Start enhancing your customer experience.</p>
                        <a href="/members">Invite</a>
                    </div>
                </div>
                <p><a href='#' onClick={hideOnboarding}>Skip this...<span className="material-symbols-outlined">skip_next</span></a></p>

              </div>
            ) }

            <div className="date-selection" >
            <div style={{padding:'5px', width:'70%'}}>
            { showInitialSetup && (
            <>
            Welcome back to our Customer Experience Platform, where we revolutionize the way you interact with your customers <br/>
              through automated,content-rich engagements on your website / applications.
              </>
                )}
            </div>
            <div onClick={handleDateFromClick}>
            <input 
                    type="date" 
                    ref={dateFromRef}
                    value={dateFrom} 
                    onClick={handleDateFromClick}                    
                    onChange={(e) => setDateFrom(e.target.value)} 
                />
              </div>  
            <div onClick={handleDateToClick}>
                <input  
                    type="date" 
                    ref={dateToRef}
                    value={dateTo} 
                    onClick={handleDateToClick}                    
                    onChange={(e) => setDateTo(e.target.value)} 
                />
              </div>  
            </div>
            <div >
            <div className="metrics">
                {/* Displaying fetched data */}
                {!projectId && (
                <div className="metric">
                <h1><span className="material-symbols-outlined">support_agent</span>{dashboardData.active_projects}</h1>
                <p>Active Agents</p>
                 </div>
                )}

                <div className="metric">
                    <h1><span className="material-symbols-outlined">chat</span>{dashboardData.total_interactions}</h1>
                    <p>Total Interactions</p>
                </div>
                {projectId && (
                <div className="metric">
                <h1><span className="material-symbols-outlined">flag</span>{dashboardData.total_goals}</h1>
                <p>Goals</p>
                 </div>
                )}

                <div className="metric">
                    <h1><span className="material-symbols-outlined">group</span>{dashboardData.total_visitors}</h1>
                    <p>Total Visitors</p>
                </div>
                <div className="metric">
                    <h1><span className="material-symbols-outlined">visibility</span>{dashboardData.total_visits}</h1>
                    <p>Total Visits</p>
                </div>
            </div>
            </div>
            <br/>
            <div className='welcome-section'>
            <h3>Daily graph of visitors Vs interactions</h3>
            <div className="visuals">
                <div className="line-graph">
                    <canvas ref={chartRef}></canvas>
                </div>
            </div>
            </div>

            <br/>
            <br/>
            <br/>
            <div className='welcome-section'>
            <h3>History of AI responses</h3>
            <div className="visuals">
                <div className="line-graph">
                    <canvas ref={barChartRef}></canvas>
                </div>
            </div>
            </div>


        </div>
    );
};

export default Dashboard;
