// src/components/pages/Dashboard/index.js
import React, { useState, useEffect } from 'react';
import './Dashboard.css';
import { useNavigate, useLocation } from 'react-router-dom';

import axios from 'axios';

function Setup({ userId: initialUserId, onClose }) {
    const [accountName, setAccountName] = useState('');
    const [title, setTitle] = useState('');
    const [industry, setIndustry] = useState('');
    const [street, setStreet] = useState('');
    const [city, setCity] = useState('');
    const [state, setState] = useState('');
    const [country, setCountry] = useState('');
    const [postalCode, setPostalCode] = useState('');
    const [email] = useState(localStorage.getItem('email'));
    const [name, setName] = useState(localStorage.getItem('name'));
    const [mobile, setMobile] = useState(localStorage.getItem('mobile'));
    const [userId] = useState(initialUserId || localStorage.getItem('user_id'));
    const [accountId] = useState(localStorage.getItem('account_id'));
    const navigate = useNavigate();
    const [isUpdating, setIsUpdating] = useState(false);
    const [errorMessage, setErrorMessage] = useState('');
    const isModal = Boolean(initialUserId);
    const [isSuccess, setSuccess] = useState(false);
    const [successMessage, setSuccessMessage] = useState('');

    useEffect(() => {
        const fetchAccountDetails = async () => {
            try {
                const response = await axios.get(`/accounts/${accountId}`);
                const accountDetails = response.data;

                // Populate your state with the account details
                setAccountName(accountDetails.account_name || '');
                setTitle(accountDetails.title || '');
                setIndustry(accountDetails.industry || '');
                setStreet(accountDetails.address?.street || '');
                setCity(accountDetails.address?.city || '');
                setState(accountDetails.address?.state || '');
                setCountry(accountDetails.address?.country || '');
                setPostalCode(accountDetails.address?.postal_code || '');

            } catch (error) {
                console.error('Error fetching account details:', error);
                // Handle error appropriately
            }
        };

        if (accountId) {
            setIsUpdating(true);
            fetchAccountDetails();
        }
    }, [accountId]);

    const validateForm = () => {
        // Simple validation. You can enhance this with a validation library if needed.
        if (!name || !mobile || !accountName || !title || !industry) {
            setErrorMessage('Please fill out all required fields.');
            return false;
        }
        if (mobile.length < 10) {
            setErrorMessage('Mobile number should be atleast 10 digits.');
            return false;
        }
        return true;
    };

    const handleSubmit = async (e) => {
        e.preventDefault();

        if (!validateForm()) {
            return;
        }

        const accountData = {
            user_id: userId,
            account_name: accountName,
            name: name,
            email: email,
            mobile: mobile,
            title: title,
            industry: industry,
            address: {
                street: street,
                city: city,
                state: state,
                country: country,
                postal_code: postalCode
            }
        };

        try {
            let response;

            if (isUpdating) {
                response = await axios.post(`/accounts/${accountId}`, accountData);                  
            } else {
                response = await axios.post('/accounts/', accountData);
            }

            if (response.request.status === 200) {
                setSuccess(true);
                setSuccessMessage('Member details has been updated!');
                setErrorMessage('');
                // Assuming response.data contains the created account data
                console.log("third account value checking",response.data)
                localStorage.setItem('account_id', response.data._id);
                localStorage.setItem('account_name', response.data.account_name);
                localStorage.setItem('account_type', response.data.account_type);
                localStorage.setItem('account_is_active', response.data.is_active);
                localStorage.setItem('name', name);
                localStorage.setItem('mobile', mobile);
                localStorage.setItem('setup_done', 1);
            } else {
                setErrorMessage('Some issue. Please try again!');
            }

        } catch (error) {
            console.error("Error setting up account:", error);
            setErrorMessage('Failed to set up account. Please try again.');
        }
    };

    const renderContent = () => (
        <div>
            {errorMessage && <p style={{ color: 'red' }}>{errorMessage}</p>}
            {successMessage && <p style={{ color: 'green' }}>{successMessage}</p>}
            <form onSubmit={handleSubmit}>
                <h3>Your Details</h3>
                <span>Your verified email is {email}</span><br />
                <input type="text" value={name} onChange={e => setName(e.target.value)} placeholder="Name" required /><br />
                <input type="text" value={mobile} onChange={e => setMobile(e.target.value)} placeholder="Mobile (ex: +1xxxxxxxxxx)" required /><br />

                <h3>Organization Details</h3>
                <input type="text" value={accountName} onChange={e => setAccountName(e.target.value)} placeholder="Organization Name" required /><br />
                <input type="text" value={title} onChange={e => setTitle(e.target.value)} placeholder="Title" required /><br />
                <select value={industry} onChange={e => setIndustry(e.target.value)} placeholder="Industry" required>
                    <option value="" disabled>Your Industry</option>
                    <option value="Tech">Tech</option>
                    <option value="Finance">Finance</option>
                    <option value="Health">Health</option>
                </select><br />
                
                <h3>Address</h3>
                <input type="text" value={street} onChange={e => setStreet(e.target.value)} placeholder="Street" /><br />
                <input type="text" value={city} onChange={e => setCity(e.target.value)} placeholder="City" /><br />
                <input type="text" value={state} onChange={e => setState(e.target.value)} placeholder="State" /><br />
                <input type="text" value={country} onChange={e => setCountry(e.target.value)} placeholder="Country" /><br />
                <input type="text" value={postalCode} onChange={e => setPostalCode(e.target.value)} placeholder="Postal Code" /><br />

                <input type="submit" value="Submit" />
            </form>
        </div>
    );

    return (
        isModal ? (
            <div className="modal-overlay">
                <div className="modal-content">
                    <h1>Member Information <a href='#' onClick={onClose}><span className="material-symbols-outlined">close</span></a></h1>
                    {isSuccess ? (
                        <>
                            <p style={{ color: 'green' }}>{successMessage}</p>
                            <p><button onClick={onClose}>Close</button></p>
                        </>
                    ) : (
                        renderContent()
                    )}
                </div>
            </div>
        ) : (        
            <div className="main-content">
                <h1>Setup Your Account</h1>
                <p>Thanks for choosing our platform to build a better experience for your customers/users. Let's do our first step!</p>
                {renderContent()}
            </div>
        )
    );
}

export default Setup;