// src/components/pages/Dashboard/index.js
import React, { useState, useEffect } from 'react';
import '../../themes/blue/Dashboard.css';
import { useNavigate, useLocation } from 'react-router-dom';
import IndustrySelect from '../../common/Modal/Industry.js'; // Adjust the import path as necessary

import axios from 'axios';

console.log('Setup component');

function Setup({ userId: initialUserId, onClose }) {

    const [accountName, setAccountName] = useState('');
    const [title, setTitle] = useState('');
    const [industry, setIndustry] = useState('');
    const [street, setStreet] = useState('');
    const [city, setCity] = useState('');
    const [state, setState] = useState('');
    const [country, setCountry] = useState('');
    const [postalCode, setPostalCode] = useState('');
    const [email] = useState(localStorage.getItem('email'));
    const [name, setName] = useState(localStorage.getItem('name'));
    const [mobile, setMobile] = useState(localStorage.getItem('mobile'));
    const [userId] = useState(initialUserId || localStorage.getItem('user_id'));
    const [accountId] = useState(localStorage.getItem('account_id'));
    const navigate = useNavigate();
    const [isUpdating, setIsUpdating] = useState(false);
    const [errorMessage, setErrorMessage] = useState('');
    const isModal = Boolean(initialUserId);
    const [isSuccess,setSuccess] = useState(false);
    const [successMessage,setSuccessMessage] = useState('');
    const roles = localStorage.getItem('roles' || 4);
    const [activeTab, setActiveTab] = useState(roles === '4' ? 'organization': 'personal'); // Default active tab
    const [isSubmitting, setIsSubmitting] = useState(false); // Track whether a submission is in progress
    const location = useLocation(); // To determine the current route
    const isMembersPage = location.pathname.includes('/members');

    useEffect(() => {
        const fetchAccountDetails = async () => {
            try {
                const response = await axios.get(`/accounts/${accountId}`);
                const accountDetails = response.data;

                // Populate your state with the account details
                setAccountName(accountDetails.account_name || '');
                setTitle(accountDetails.title || '');
                setIndustry(accountDetails.industry || '');
                setStreet(accountDetails.address?.street || '');
                setCity(accountDetails.address?.city || '');
                setState(accountDetails.address?.state || '');
                setCountry(accountDetails.address?.country || '');
                setPostalCode(accountDetails.address?.postal_code || '');

            } catch (error) {
                console.error('Error fetching account details:', error);
                // Handle error appropriately
            }
        };

        if (accountId) {
            setIsUpdating(true);
            fetchAccountDetails();
        }
    }, [accountId]);

    const validateForm = () => {
        let missingFields = [];
        let isValid = true;
    
        if (!accountName) missingFields.push("Account Name");
        if (!title) missingFields.push("Title");
        if (!industry) missingFields.push("Industry");
        if (!name) missingFields.push("Name");
        if (!mobile) missingFields.push("Mobile");
        
        if (mobile && mobile.length < 10 && mobile.length > 15) {
            missingFields.push("Mobile number should be atleast 10 chars, max 15 including special chars.");
        }
    
        // Set error message based on missing fields
        if (missingFields.length > 0) {
            setErrorMessage(`Please fill out the following required fields: ${missingFields.join(", ")}.`);
            isValid = false;
        } else {
            setErrorMessage('');
        }
    
        // Set active tab to personal only when accountName, title, and industry are filled, but name or mobile is missing
        if (accountName && title && industry && (!name || !mobile)) {
            setActiveTab('personal');
        }
    
        return isValid;
    };
    
    const handleSubmit = async (e) => {
        e.preventDefault();
        if (isSubmitting) return; // Prevent multiple submissions

        if (!validateForm()) {
            return;
        }
        const accountData = {
            user_id: userId,
            account_name: accountName,
            roles: roles,
            name: name,
            email: email,
            mobile: mobile,
            title: title,
            industry: industry,
            address: {
                street: street,
                city: city,
                state: state,
                country: country,
                postal_code: postalCode
            }
        };

        try {
            let response;
            setIsSubmitting(true); // Re-enable the submit button

            if (isUpdating) {
                if(accountId){
                    //console.log(accountData);
                    response = await axios.post(`/accounts/${accountId}`, accountData);                  
                }else{
                    return;
                }
            } else {
                response = await axios.post('/accounts/', accountData);
            }
            
            if (response.request.status === 200) {
                setSuccess(true);
                setSuccessMessage('Member details has been updated!');
                setErrorMessage('');
                if(!accountId){
                    localStorage.setItem('roles', 2);
                }
                // Assuming response.data contains the created account data
                console.log("second check the account values",response.data)
                localStorage.setItem('account_id', response.data._id);
                localStorage.setItem('account_name', response.data.account_name);
                localStorage.setItem('account_type', response.data.account_type);
                localStorage.setItem('account_is_active', response.data.is_active);
                localStorage.setItem('name', name);
                localStorage.setItem('mobile', mobile);
                localStorage.setItem('setup_done', 1);

                const referenceURL = localStorage.getItem('referenceURL');
                const message = isUpdating 
                    ? 'Your account has been successfully updated.' 
                    : 'Welcome to our AI Platform. Your account has been successfully created. Please build an ';
                
                let targetURL = '';

                const encodedMessage = encodeURIComponent(message);
                if (isUpdating) {
                    if (parseInt(roles) === 4){
                        targetURL = `/ai.workers?message=${encodedMessage}`;
                    }else{
                        targetURL = `/dash?message=${encodedMessage}`;
                    }
                } else {
                    console.log("check the reference url values:",referenceURL)
                    console.log("check the encoded message values",encodedMessage)
                    targetURL = referenceURL ? `${referenceURL}?message=${encodedMessage}` : `/billing?message=new`;
                  //  targetURL = referenceURL ? `${referenceURL}?message=${encodedMessage}` : `/agents?message=new`;


                }           
                //const targetURL = referenceURL ? `${referenceURL}?message=${encodedMessage}` : `/agents?message=new`;
                navigate(targetURL);
                //
            }else{
                setIsSubmitting(false); // Re-enable the submit button            
                setErrorMessage('Some issue. Please try again!');
            }

        } catch (error) {
            console.error("Error setting up account:", error);
            setErrorMessage('Failed to set up account. Mobile might be already in Use with another account, Please try some other.');
            setIsSubmitting(false); // Re-enable the submit button

            // Handle error
        }
    };
    const handleIndustryChange = (event) => {
        setIndustry(event.target.value);
      };
    
      const renderOrganizationContent = () => (
        <div className='formdiv'>
            <h3>Your Organization</h3>
            <label> Name</label>
            <input type="text" value={accountName} onChange={e => setAccountName(e.target.value)} placeholder=" Name of the Organization" required /><br />
            <label> Title </label>
            <input type="text" value={title} onChange={e => setTitle(e.target.value)} placeholder="Your Title in Organization" required /><br />
            <IndustrySelect
                selectedIndustry={industry}
                onChange={handleIndustryChange}
            />
            {location.pathname !== '/setup' && (
                <>
                    <br />
                    <h3>Organization Address</h3>
                    <p>(Optional fields)</p>
                    <input type="text" value={street} onChange={e => setStreet(e.target.value)} placeholder="Street" /><br />
                    <input type="text" value={city} onChange={e => setCity(e.target.value)} placeholder="City" /><br />
                    <input type="text" value={state} onChange={e => setState(e.target.value)} placeholder="State" /><br />
                    <input type="text" value={country} onChange={e => setCountry(e.target.value)} placeholder="Country" /><br />
                    <input type="text" value={postalCode} onChange={e => setPostalCode(e.target.value)} placeholder="Postal Code" /><br />
                </>
            )}
        </div>
    );
    
    const renderPersonalContent = () => (
        <div className='formdiv'>
                <span>Your verified email is <a href={`mailto:${email}`}>{email}</a></span><br /><br/>
                <input type="text" value={name} onChange={e => setName(e.target.value)} placeholder="Name" required /><br />
                <input type="text" value={mobile} maxLength={15} onChange={e => setMobile(e.target.value)} placeholder="Mobile (ex: +1xxxxxxxxxx)" required /><br />
        </div>
    );

    const renderTabs = () => {
        // Only show both tabs if roles is 2
        return roles ? (
            <div className="tabs-container">
            <div className="tab-links">
                {roles === 2 && (
                <a href='#' id='org' onClick={() => setActiveTab('organization')} className={activeTab === 'organization' ? 'activated' : ''}>Organization</a>
                )}
                <a href='#' id='personal' onClick={() => setActiveTab('personal')} className={activeTab === 'personal' ? 'activated' : ''}>Personal</a>
                {!isMembersPage && (
                    <a href='/logout' id='logout'>Logout</a>
                )}
            </div>
            </div>
        ) : null;
    };

    const renderContentForSetup = () => {
        // Combined content for /setup without tabs
        return (
            <>
                {renderPersonalContent()}
                {renderOrganizationContent()}
            </>
        );
    };

    const renderContentForSettings = () => {
        // Content for /settings with tabs
        return (
            <>
                {renderTabs()}
                {activeTab === 'personal' ? renderPersonalContent() : renderOrganizationContent()}
            </>
        );
    };


    const renderContent = () => {
        // Determine the content based on the current route
        return location.pathname === '/setup' ? renderContentForSetup() : renderContentForSettings();
    };


    return (
        isModal ? (
            <div className="modal-overlay">
                <div className="modal-content">
                    <h1>Member Information</h1>
                    <a href="#" onClick={onClose} className='closeontopright'><span className="material-symbols-outlined">cancel</span></a>

                    {isSuccess ? (
                        <>
                            <p style={{ color: 'green' }}>{successMessage}</p>
                            <p><button onClick={onClose}>Close</button></p>
                        </>
                    ) : (
                        <form onSubmit={handleSubmit}>
                        {renderContent()}

                        <button onClick={handleSubmit} className="button" disabled={isSubmitting}>
                         {isSubmitting ? 'Updating ...' : 'Continue'}
                        </button>

                        </form>

                    )}
                </div>
            </div>
        ) : (        
            <div className="dashboard-container">
            <h2>{location.pathname === '/setup' ? 'Setup Your Account' : 'Settings'}</h2>
            <p>{location.pathname === '/setup'
                ? "We're delighted you've chosen the thought platform to enhance your customer experience. Our journey begins here, and we're excited to help you make a significant impact on your customers and end-users. "
                : "Feel free to update your settings here."}</p>
                {errorMessage && <p className='error'>{errorMessage}</p>}
                {successMessage && <p className='success'>{successMessage}</p>}
                {!accountId && (
                <p className='success'>
                    Thank you for signing up with thought's Platform. You're almost there! Please complete your setup by updating your organization and personal information.
                </p>
                )}

                <form onSubmit={handleSubmit}>
                {renderContent()}
                <br/><br/>
                <button onClick={handleSubmit} className="button" disabled={isSubmitting}>
                         {isSubmitting ? 'Updating ...' : 'Submit'}
                </button>

                </form>
            </div>
        )
    );
}

export default Setup;