import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import CommonTaskList from '../../common/CommonTaskList';
import TaskBar from '../../common/TaskListBar';


const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'tasks', label: 'Tasks', icon: 'assignment' },
  { id: 'transactions', label: 'Transactions', icon: 'payments' },
  { id: 'info', label: 'Info', icon: 'info' },
  { id: 'users', label: 'Users', icon: 'people' }
];


const CustomerView = ({ customerId, onClose }) => {
  const [customerData, setCustomerData] = useState(null);
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);


  useEffect(() => {
    const fetchData = async () => {
      try {
        const customerResponse = await axios.get(`/customers/${customerId}`);
        setCustomerData(customerResponse.data);


        if (customerResponse.data) {
          try {
            const transactionsResponse = await axios.get(`/transactions?customer_id=${customerId}`);
            setTransactions(transactionsResponse.data || []);
          } catch (transactionsError) {
            console.error("Error fetching transactions:", transactionsError);
            setTransactions([]);
          }
        }


        setLoading(false);
      } catch (err) {
        console.error("Error fetching customer details:", err);
        setError('Error fetching customer details.');
        setLoading(false);
      }
    };


    fetchData();
  }, [customerId]);


  const renderTabContent = () => {
    if (!customerData) return null;
    console.log("vijay",customerData)


    switch (activeTab) {
      case 'dashboard':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '12px',
                color: '#333' // Black color for header icon
              }}>
                dashboard
              </span>
              Customer Summary
            </h4>


            <p><strong>Phone:</strong> {customerData.phone}</p>
            <p><strong>Status:</strong> {customerData.status || 'Active'}</p>
            <p><strong>Created:</strong> {new Date(customerData.created_date).toLocaleDateString()}</p>
            <p><strong>Last Activity:</strong> {new Date(customerData.updated_date).toLocaleDateString()}</p>


            <TaskBar customerId={customerId} />
            <h4>Recent Activity</h4>


            <CommonTaskList
              presetFilters={{ customer_id: customerId }}
              rowsPerPage={2}
              initialSearchQuery=""
            />
          </div>
        );


      case 'tasks':
        return (
          <div style={{ marginTop: '15px' }}>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333' // Black color for header icon
              }}>
                assignment
              </span>
              Customer Tasks
            </h4>
            <div style={{
              border: '1px solid #eee',
              borderRadius: '8px',
              padding: '15px',
              marginTop: '10px'
            }}>
              <CommonTaskList
                presetFilters={{ customer_id: customerId }}
                rowsPerPage={5}
                initialSearchQuery=""
              />
            </div>
          </div>
        );


      case 'transactions':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333' // Black color for header icon
              }}>
                payments
              </span>
              Customer Transactions
            </h4>
            {transactions.length > 0 ? (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                <table style={{ width: '100%', borderCollapse: 'collapse' }}>
                  <thead>
                    <tr style={{ borderBottom: '1px solid #eee' }}>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Date</th>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Amount</th>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Type</th>
                      <th style={{ textAlign: 'left', padding: '8px' }}>Status</th>
                    </tr>
                  </thead>
                  <tbody>
                    {transactions.map((transaction, index) => (
                      <tr key={index} style={{ borderBottom: '1px solid #eee' }}>
                        <td style={{ padding: '8px' }}>{new Date(transaction.date).toLocaleDateString()}</td>
                        <td style={{ padding: '8px' }}>${transaction.amount}</td>
                        <td style={{ padding: '8px' }}>{transaction.type}</td>
                        <td style={{ padding: '8px' }}>{transaction.status}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '20px',
                marginTop: '10px',
                textAlign: 'center',
                color: '#666'
              }}>
                No transactions found for this customer
              </div>
            )}
          </div>
        );


     case 'info':
  return (
    <div>
      <h4>
        <span className="material-symbols-outlined" style={{
          verticalAlign: 'middle',
          marginRight: '8px',
          color: '#333' // Black color for header icon
        }}>
          info
        </span>
        Customer Information
      </h4>

      <div style={{
        border: '1px solid #eee',
        borderRadius: '8px',
        padding: '15px',
        marginTop: '10px'
      }}>
        <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '15px' }}>
          <div>
            <p><strong>First Name:</strong> {customerData.first_name}</p>
            <p><strong>Last Name:</strong> {customerData.last_name}</p>
            <p><strong>Email:</strong> {customerData.email || 'N/A'}</p>
            <p><strong>Phone:</strong> {customerData.phone}</p>
          
            <p><strong>Street:</strong> {customerData.address?.street || 'N/A'}</p>
            <p><strong>City:</strong> {customerData.address?.city || 'N/A'}</p>
            <p><strong>State:</strong> {customerData.address?.state || 'N/A'}</p>
            <p><strong>Postal Code:</strong> {customerData.address?.postal_code || 'N/A'}</p>
            <p><strong>Country:</strong> {customerData.address?.country || 'N/A'}</p>
          </div>
        </div>

        <div style={{ marginTop: '15px' }}>
          <p><strong>Customer Type:</strong> {customerData.customer_type || 'Standard'}</p>
          <p><strong>Account Created:</strong> {new Date(customerData.created_date).toLocaleString()}</p>
          <p><strong>Last Updated:</strong> {new Date(customerData.updated_date).toLocaleString()}</p>
          <p><strong>Last Login:</strong> {customerData.last_login ? new Date(customerData.last_login).toLocaleString() : 'Never'}</p>
        </div>
      </div>
    </div>
  );


      case 'users':
        return (
          <div>
            <h4>
              <span className="material-symbols-outlined" style={{
                verticalAlign: 'middle',
                marginRight: '8px',
                color: '#333' // Black color for header icon
              }}>
                people
              </span>
              Customer Users
            </h4>
            {customerData.users?.length > 0 ? (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '15px',
                marginTop: '10px'
              }}>
                {customerData.users.map((user, index) => (
                  <div key={index} style={{
                    padding: '10px',
                    borderBottom: index < customerData.users.length - 1 ? '1px solid #eee' : 'none'
                  }}>
                    <p><strong>{user.name}</strong> ({user.role})</p>
                    <p>Email: {user.email}</p>
                    <p>Last Active: {user.last_active ? new Date(user.last_active).toLocaleString() : 'Never'}</p>
                  </div>
                ))}
              </div>
            ) : (
              <div style={{
                border: '1px solid #eee',
                borderRadius: '8px',
                padding: '20px',
                marginTop: '10px',
                textAlign: 'center',
                color: '#666'
              }}>
                No additional users found for this customer
              </div>
            )}
          </div>
        );


      default:
        return null;
    }
  };


  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading customer data...</div>
      </Modal>
    );
  }


  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }


  const visibleTabs = TAB_CONFIG.filter(tab =>
    tab.id !== 'users' || (customerData && customerData.customer_type === 'enterprise')
  );


  return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          gap: '20px',
          padding: '0px 10px 10px',
          borderBottom: '1px solid #e0e0e0'
        }}>
          <div style={{
  width: '60px',
  height: '60px',
  borderRadius: '50%',
  backgroundColor: '#e0e0e0', // Light grey color
  color: '#000', // Black font color
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
  fontSize: '20px',
  fontWeight: 'bold'
}}>
            {customerData.first_name?.charAt(0)?.toUpperCase() || 'C'}
          </div>
          <div>
            <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '20px', color: 'black' }}>
              {customerData.first_name} {customerData.last_name}
            </h3>
            <p style={{  marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
              ID : #{customerData.customer_id}
            </p>
          </div>
        </div>


        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '20px +6px'
        }}>
          {visibleTabs.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '2px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: activeTab === tab.id ? '100px' : '50px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-20px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>


        {/* Content */}
        <div style={{ padding: '0 20px', overflow: 'hidden' }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};


export default CustomerView;