import React, { useState, useEffect } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import CustomerView from './CustomerView';
import CustomerUpdate from './CustomerUpdate';
import CustomerDB from './CustomerDB';
import CommonData from '../../common/CommonData';

const CUSTOMERS_HEADER_TO_KEY_MAP = {
  "First Name": { key: "first_name", label: "First Name", sortable: true },
  "Last Name": { key: "last_name", label: "Last Name", sortable: false },
  "Phone": { key: "phone", label: "Phone", sortable: false },
  "Email": { key: "email", label: "Email", sortable: false },
  "Customer Type": { key: "customer_type", label: "Customer Type", sortable: false },
  "Created Date": { key: "created_date", label: "Created Date", sortable: true },
};

const CustomerList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const [accountId, setAccountId] = useState(null);
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery,
    created_date_from: '',
    created_date_to: '',
    customer_type: '',
    status: ''
  });
  const [fetchUrl, setFetchUrl] = useState('');
  const [selectedCustomer, setSelectedCustomer] = useState(null);
  const [dateError, setDateError] = useState('');

  const enums = CommonData();

  useEffect(() => {
    const storedAccountId = localStorage.getItem('account_id');
    if (storedAccountId) {
      setAccountId(storedAccountId);
    }
  }, []);

  useEffect(() => {
    if (accountId) {
      setFetchUrl(buildFetchUrl(searchParams));
    }
  }, [searchParams, accountId]);

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'created_date_from',
      type: 'date',
      placeholder: 'From Date',
      defaultValue: '',
      max: searchParams.created_date_to || undefined,
      onChange: (e) => handleDateChange('created_date_from', e.target.value)
    },
    {
      name: 'created_date_to',
      type: 'date',
      placeholder: 'To Date',
      defaultValue: '',
      min: searchParams.created_date_from || undefined,
      onChange: (e) => handleDateChange('created_date_to', e.target.value)
    },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ]
    },
    {
      name: 'customer_type',
      type: 'dropdown',
      placeholder: 'Customer Type',
      options: [
        { label: 'All', value: '' },
        { label: 'Individual', value: 'individual' },
        { label: 'Enterprise', value: 'enterprise' }
      ]
    },
  ];

  const actionFormats = [
    { label: "view", modal: true, url: "/customers/{id}" },
    { label: "edit", modal: true, url: "/customers/{id}" },
    { label: "delete", modal: true, url: "/customers/{id}" }
  ];

  const handleActionClick = (customerId, action) => {
    if (["view", "edit", "delete"].includes(action.label)) {
      setSelectedCustomer({ customerId, modalType: action.label });
    }
  };

  const handleDateChange = (field, value) => {
    setSearchParams(prev => {
      const newParams = {
        ...prev,
        [field]: value
      };
      
      // Clear the error when either date changes
      if (dateError) {
        setDateError('');
      }
      
      return newParams;
    });
  };

  const validateDates = (fromDate, toDate) => {
    if (fromDate && toDate && new Date(toDate) < new Date(fromDate)) {
      return "To Date must be greater than or equal to From Date";
    }
    return "";
  };

  const handleSearch = (params) => {
    const error = validateDates(params.created_date_from, params.created_date_to);
    setDateError(error);
    
    if (error) {
      return;
    }

    setSearchParams({
      ...params,
      skip: 0,
      limit: rowsPerPage
    });
  };

  const handleClearFilters = () => {
    setSearchParams({
      q: '',
      created_date_from: '',
      created_date_to: '',
      customer_type: '',
      status: '',
      skip: 0,
      limit: rowsPerPage
    });
    setDateError('');
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  const buildFetchUrl = (params) => {
    if (!accountId) return '';

    let url = `/customers/?account_id=${accountId}&skip=${params.skip}&limit=${params.limit}`;
    if (params.q) url += `&q=${encodeURIComponent(params.q)}`;
    
    // Date filtering logic
    if (params.created_date_from) {
      url += `&from_date=${params.created_date_from}`;
      
      // Only add to_date if it's provided and valid
      if (params.created_date_to && params.created_date_to >= params.created_date_from) {
        url += `&to_date=${params.created_date_to}`;
      }
    } 
    // If only to_date is provided (without from_date)
    else if (params.created_date_to) {
      url += `&to_date=${params.created_date_to}`;
    }
    
    if (params.customer_type) url += `&customer_type=${params.customer_type}`;
    if (params.status) url += `&status=${params.status}`;
    return url;
  };

  const closeModal = () => setSelectedCustomer(null);

  if (!accountId || !fetchUrl) return <div>Loading...</div>;

  return (
    <div className="customer-list-container" style={{ overflow: 'visible', height: 'auto' }}>

      {dateError && (
        <div className="error-message" style={{ color: 'red', marginBottom: '10px' }}>
          {dateError}
        </div>
      )}
      
      <SearchPage
        fetchDataUrl={fetchUrl}
        ListComponent={List}
        itemKeyMapping={CUSTOMERS_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        searchFields={searchFields}
        onSearch={handleSearch}
        onPageChange={handlePageChange}
        onClearFilters={handleClearFilters}
      />

      {selectedCustomer?.modalType === "view" && (
        <CustomerView customerId={selectedCustomer.customerId} onClose={closeModal} />
      )}
      {selectedCustomer?.modalType === "edit" && (
        <CustomerUpdate customerId={selectedCustomer.customerId} onClose={closeModal} />
      )}
      {selectedCustomer?.modalType === "delete" && (
        <CustomerDB customerId={selectedCustomer.customerId} onClose={closeModal} />
      )}
    </div>
  );
};

export default CustomerList;