import React, { useEffect, useState } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';
import { useNavigate } from 'react-router-dom';






const customerSchema = {
  type: "object",
  required: ["first_name", "last_name", "email", "phone", "customer_type"],
  properties: {
    account_id: { type: "string", title: "Account ID" },
    user_id: { type: "string", title: "User ID" },
    first_name: { type: "string", title: "First Name" },
    last_name: { type: "string", title: "Last Name" },
    email: { type: "string", format: "email", title: "Email" },
    phone: { type: "string", title: "Phone" },
    address: {
      type: "object",
      title: "Address",
      required: ["street", "city", "state", "postal_code", "country"],
      properties: {
        street: { type: "string", title: "Street" },
        city: { type: "string", title: "City" },
        state: { type: "string", title: "State" },
        postal_code: { type: "string", title: "Postal Code" },
        country: { type: "string", title: "Country" }
      }
    },
    customer_type: {
      type: "string",
      title: "Customer Type",
      enum: ["individual", "enterprise"],
      enumNames: ["Individual", "Enterprise"],
      default: "individual"
    }
  },
  dependencies: {
    customer_type: {
      oneOf: [
        {
          properties: {
            customer_type: { const: "individual" }
          }
        },
        {
          properties: {
            customer_type: { const: "enterprise" },
            company_name: { type: "string", title: "Company Name" },
            contact_person: { type: "string", title: "Contact Person" },
            company_email: {
              type: "string",
              format: "email",
              title: "Company Email"
            },
            company_phone: {
              type: "string",
              title: "Company Phone"
            },
            billing_address: {
              type: "object",
              title: "Billing Address",
              required: ["street", "city", "state", "postal_code", "country"],
              properties: {
                street: { type: "string", title: "Street" },
                city: { type: "string", title: "City" },
                state: { type: "string", title: "State" },
                postal_code: { type: "string", title: "Postal Code" },
                country: { type: "string", title: "Country" }
              }
            }
          },
          required: ["company_name", "contact_person"]
        }
      ]
    }
  }
};




const uiSchema = {
  account_id: { "ui:widget": "hidden" },
  user_id: { "ui:widget": "hidden" },
  first_name: { "ui:placeholder": "Enter first name" },
  last_name: { "ui:placeholder": "Enter last name" },
  email: { "ui:placeholder": "Enter email" },
  phone: { "ui:placeholder": "Enter phone number" },
  address: {
    "ui:field": "GridLayout",
    "ui:options": {
      rowGap: "10px"
    },
    street: { "ui:placeholder": "Enter street address" },
    city: { "ui:placeholder": "Enter city" },
    state: { "ui:placeholder": "Enter state/province" },
    postal_code: { "ui:placeholder": "Enter postal code" },
    country: { "ui:placeholder": "Enter country" }
  },
  customer_type: {
    "ui:widget": "radio",
    "ui:options": {
      inline: true
    }
  },
  company_email: { "ui:placeholder": "Company email" },
  company_phone: { "ui:placeholder": "Company phone number" },
  billing_address: {
    "ui:field": "GridLayout",
    "ui:options": {
      rowGap: "10px"
    },
    street: { "ui:placeholder": "Enter billing street address" },
    city: { "ui:placeholder": "Enter billing city" },
    state: { "ui:placeholder": "Enter billing state/province" },
    postal_code: { "ui:placeholder": "Enter billing postal code" },
    country: { "ui:placeholder": "Enter billing country" }
  }
};




const transformErrors = () => [];




const CustomerForm = ({ onSubmit, initialData }) => {
  const [formData, setFormData] = useState({
    customer_type: "individual",
    ...initialData
  });




  const [currentSchema, setCurrentSchema] = useState(customerSchema);
  const navigate = useNavigate();




  useEffect(() => {
    const newSchema = JSON.parse(JSON.stringify(customerSchema));
    if (formData.customer_type === "enterprise") {
      newSchema.properties.company_name = { type: "string", title: "Company Name" };
      newSchema.properties.contact_person = { type: "string", title: "Contact Person" };
      newSchema.properties.company_email = { type: "string", format: "email", title: "Company Email" };
      newSchema.properties.company_phone = { type: "string", title: "Company Phone" };
      newSchema.properties.billing_address = {
        type: "object",
        title: "Billing Address",
        required: ["street", "city", "state", "postal_code", "country"],
        properties: {
          street: { type: "string", title: "Street" },
          city: { type: "string", title: "City" },
          state: { type: "string", title: "State" },
          postal_code: { type: "string", title: "Postal Code" },
          country: { type: "string", title: "Country" }
        }
      };
    } else {
      delete newSchema.properties.company_name;
      delete newSchema.properties.contact_person;
      delete newSchema.properties.company_email;
      delete newSchema.properties.company_phone;
      delete newSchema.properties.billing_address;
    }
    setCurrentSchema(newSchema);
  }, [formData.customer_type]);




const handleSubmit = async ({ formData }) => {
  try {
    await onSubmit(formData); // If onSubmit is async
    navigate('/customers');   // Redirect after success
  } catch (error) {
    console.error("Submission failed:", error);
  }
};




  return (
    <div style={{ maxWidth: "600px", paddingLeft: "20px" }}>
  <Form
    schema={currentSchema}
    uiSchema={uiSchema}
    formData={formData}
    onChange={({ formData }) => {
      if (formData.customer_type === "individual") {
        formData.company_name = undefined;
        formData.contact_person = undefined;
        formData.company_email = undefined;
        formData.company_phone = undefined;
        formData.billing_address = undefined;
      }
      setFormData(formData);
    }}
    onSubmit={handleSubmit}
    validator={validator}
    transformErrors={transformErrors}
    showErrorList={false}
   
templates={{
  ButtonTemplates: {
    SubmitButton: (props) => {
      const [isHovered, setIsHovered] = React.useState(false);
     
      return (
        <div style={{ display: 'flex', justifyContent: 'flex-start', marginTop: '20px' }}>
          <button
            type="submit"
            className="btn"
            disabled={props.disabled}
            style={{
              padding: '10px 16px',
              borderRadius: '4px',
              backgroundColor: '#f0f0f0',
              backgroundColor: isHovered ? '#00D563' : '#f0f0f0',
              color: isHovered ? 'white' : '#00D563',
              border: '1px solid white',
              cursor: props.disabled ? 'not-allowed' : 'pointer',
              transition: 'all 0.3s ease',
              opacity: props.disabled ? 0.7 : 1
            }}
            onMouseEnter={() => !props.disabled && setIsHovered(true)}
            onMouseLeave={() => !props.disabled && setIsHovered(false)}
          >
            {props.children || 'Submit'}
          </button>
        </div>
      );
    }
  }
}}
  />
</div>
  );
};




export default CustomerForm;