import React, { useState, useEffect } from 'react';
import { useLocation } from 'react-router-dom';
import WebPricing from './Pricing';
import Payments from './Payments.js'; // Assuming you have a component for payments
import Tabs from '../../common/Tabs';
import axios from 'axios'; // Assuming axios for HTTP requests

const Billing = () => {
  const location = useLocation();
  const [isPaymentRequired, setIsPaymentRequired] = useState(false);
  const [selectedPlan, setSelectedPlan] = useState(null);
  const [paymentStatus, setPaymentStatus] = useState(null); // New state for tracking payment status
  const [activeTab, setActiveTab] = useState('plans'); // Initialized to 'plans'
  const subscriptionId = localStorage.getItem('subscription_id') || ""; // Initialized to 'plans'

  const handlePlanSelection = (plan) => {
    setSelectedPlan(plan);
    setIsPaymentRequired(plan.price > 0);
    setActiveTab('plans'); // Ensure user is always brought back to the plans tab upon selecting a plan
  };

  useEffect(() => {
    const handleRouteChange = () => {
      const currentPath = location.pathname;
      //console.log(currentPath);
      const queryParams = new URLSearchParams(location.search);
      const sessionId = queryParams.get('session_id');
      const isCancel = currentPath.includes('cancel');
      const isSuccess = currentPath.includes('success');
  
      if (sessionId && isSuccess) {
        // Handle success case
        console.log('Handle success', sessionId);
        // Optionally verify session with your backend
        verifyPaymentSession(sessionId);
      } else if (isCancel) {
        // Handle cancellation
        console.log('Handle cancellation');
        setPaymentStatus('cancelled');
      }
    };
  
    handleRouteChange();
  }, [location]);
  
  const verifyPaymentSession = async (sessionId) => {
    try {
      const response = await axios.post('/saas/stripe/verify', { session_id: sessionId });
      if (response.data.success) {
        localStorage.setItem('subscription_status', response.data.status); //setting up the
        setPaymentStatus('success');
      } else {
        setPaymentStatus('failure');
      }
      setActiveTab('plans'); // Direct user to 'plans' tab on payment verification
    } catch (error) {
      console.error('Error verifying payment session:', error);
      setPaymentStatus('failure');
      setActiveTab('plans'); // Direct user to 'plans' tab on error
    }
  };

  const handleManageBilling = async () => {
    try {
      const response = await axios.post('/saas/stripe/portal/', { subscription_id: subscriptionId });
      //https://stripe.com/docs/customer-management
      
      const portalUrl = response.data.url;
      window.location.href = portalUrl; // Redirect to Stripe Customer Portal
    } catch (error) {
      console.error('Error creating portal session:', error);
      // Handle error (e.g., show error message)
    }
  };

  const tabsData = [
    { id: 'plans', label: 'Plans' },
    { id: 'payments', label: 'Payments' },
    { id: 'pay', label: 'Payment Modes' },
  ];

  return (
    <div className="dashboard-container">
      <h2>Subscription</h2>
      <p>
      Welcome to the Subscription and Billing Management section. Here, you can browse our range of subscription plans tailored to fit various needs and scales.
      </p>
      {paymentStatus === 'success' && <p className='success'>Payment is successful. Your subscription bas been activated!</p>}
      {paymentStatus === 'cancelled' && <p className='error'>Payment cancelled or failed, please make the payment to activate your account.</p>}
      {paymentStatus === 'failure' && <p className='error'>There was an issue with your payment.</p>}

      <Tabs activeTab={activeTab} setActiveTab={setActiveTab} tabs={tabsData}>
        {activeTab => {
          switch(activeTab) {
            case 'plans':
              return <WebPricing onPlanSelect={handlePlanSelection} />;
            case 'payments':
              return <Payments />;
            case 'pay':
              //return <Payment selectedPlan={selectedPlan} />;
              if(subscriptionId){
                handleManageBilling();
                return 'Please wait while we redirecting you to the payment page...';  
              }else{
                return "You don't have any active subcriptions !. Please subscribe...";
              }
            default:
              return null;
          }
        }}
      </Tabs>
    </div>
  );
};

export default Billing;
