import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useLocation } from 'react-router-dom';

const WebPricing = () => {
    const [subscriptionTypes, setSubscriptionTypes] = useState([]);
    const [universalAgentCount, setUniversalAgentCount] = useState(parseInt(localStorage.getItem('active_agents') || 1));
    const userId = localStorage.getItem('user_id');
    const accountId = localStorage.getItem('account_id');
    const [message, setMessage] = useState('');
    const [messageType, setMessageType] = useState('');
    const location = useLocation();
    const isBillingPage = location.pathname.startsWith('/billing');
    const [showFeatures, setShowFeatures] = React.useState(false);

    const subscriptionId = localStorage.getItem('subscription_id') || '';
    const [subscriptionType, setSubscriptionType] = useState(localStorage.getItem('subscription_type') || -1);
    const [activeAgents, setActiveAgents] = useState(parseInt(localStorage.getItem('active_agents') || 0));
    const [subscriptionAgents, setSubscriptionAgents] = useState(parseInt(localStorage.getItem('subscription_agents') || 0));
    const [subscriptionStatus, setSubscriptionStatus] = useState(localStorage.getItem('subscription_status') || "");
    const [stripeSubscriptionId, setStripeSubscriptionId] = useState(localStorage.getItem('stripe_subscription_id') || null);

    useEffect(() => {
        const fetchSubscriptionTypes = async () => {
            try {
                const response = await axios.get('/saas/subscription-types/');
                setSubscriptionTypes(response.data);
                // Following code is when user is clicked subscribe without signin. Check if there's data from a redirect after signup
                const storedPlanId = localStorage.getItem('planId');
                const storedAgentCount = localStorage.getItem('agentCount');

                if (storedPlanId && storedAgentCount) { 
                    setSubscriptionType(storedPlanId);
                    setUniversalAgentCount(parseInt(storedAgentCount));
                    localStorage.removeItem('planId');
                    localStorage.removeItem('agentCount');
                }

            } catch (error) {
                console.error('There was an error fetching the subscription types', error);
            }
        };

        fetchSubscriptionTypes();
    }, []);


    useEffect(() => {
        const fetchSubscriptionDetails = async () => {
            if (subscriptionId) {
                try {
                    const response = await axios.get(`/saas/subscription/${subscriptionId}`);
                    const subscriptionData = response.data;                    
                    setSubscriptionType(subscriptionData.subscription_type_id);
                    setSubscriptionAgents(subscriptionData.subscription_agents_count);
                    setUniversalAgentCount(subscriptionData.subscription_agents_count);
                    setActiveAgents(subscriptionData.active_agents_count);
                    setSubscriptionStatus(subscriptionData.status);
                    setStripeSubscriptionId(subscriptionData.stripe_subscription_id);
                    localStorage.setItem('subscription_status', subscriptionData.status); //setting up the Updated Status (If any changes)

                } catch (error) {
                    console.error('Error fetching subscription details:', error);
                }
            }
        };

        fetchSubscriptionDetails();
    }, [subscriptionId]);
    
    const incrementAgentCount = () => {
        setUniversalAgentCount(prevCount => prevCount + 1);
    };
    
    const decrementAgentCount = () => {
        setUniversalAgentCount(prevCount => Math.max(prevCount - 1, activeAgents, 1));
    };
            
    const renderPrice = (type) => {
        //const count = agentCounts[type.subscription_type_id] || 0;
        const total = universalAgentCount * type.price;
    
        if (type.price === -1) {
            return <span>Contact for pricing</span>;
        }
        return total === 0 ? <span>$0 - It's Free</span> : <span>${total}/month (total)</span>;
    };
    
    const renderFeatures = (features) => {
        return (
            <ul>
                {features.map((feature, index) => (
                    <li key={index}> {feature}</li>
                ))}
            </ul>
        );
    };

    const handleSubscribe = async (type) => {
        //const agentCount = agentCounts[type.subscription_type_id] || 0;
        const agentCount = universalAgentCount;
        const totalCost = agentCount * type.price;
        const perAgentCost = type.price; // Make sure this is the correct calculation
        if(!userId){
            localStorage.setItem('referenceURL', '/billing');
            localStorage.setItem('planId', type.subscription_type_id.toString());
            localStorage.setItem('agentCount', agentCount.toString());
            
            window.location.href = `/signup`;
                return;                
        }

        try {
            const payload = {
                user_id: userId,
                account_id: accountId,
                subscription_type_id: type.subscription_type_id,
                subscription_agents_count: agentCount,
                active_agents_count: activeAgents, // Initially, active agents might be 0
                peragent_cost: parseFloat(perAgentCost),
                total_cost: parseFloat(totalCost),
            };
            //console.log(payload);
            //return;
            const response = await axios.post('/saas/subscribe', payload);
            const subscriptionData = response.data.subscription;
            const checkoutURL = response.data.checkout_url;

            setMessage('New Subscription has been created successfully.');
            setMessageType('success');
            // Update local storage with new subscription details
            localStorage.setItem('subscription_id', subscriptionData.subscription_id);
            localStorage.setItem('subscription_type', subscriptionData.subscription_type_id);
            localStorage.setItem('subscription_agents', subscriptionData.subscription_agents_count);
            localStorage.setItem('active_agents', subscriptionData.active_agents_count);
            localStorage.setItem('subscription_status', subscriptionData.status);
            // Update state as well
            setSubscriptionType(subscriptionData.subscription_type_id);
            setSubscriptionAgents(subscriptionData.subscription_agents_count);
            setActiveAgents(subscriptionData.active_agents_count);
            setSubscriptionStatus(subscriptionData.status);

            if(checkoutURL){
                window.location.href = checkoutURL;
            }
            // ... Update other subscription details as needed
        } catch (error) {
            setMessage('Error subscribing: ' + error.message);
            setMessageType('error');
        }
    };

    const handleContactUs = () => {
        // Redirect to the contact page with pre-filled information
        // Assuming you have a route like '/contact?email=...'
        // and a form on the contact page that can handle these query parameters
        const contactUrl = `/contact`; //?email=${userEmail}&name=${userName}&companyName=${companyName}&agentType=${subscriptionType}&agentCount=${activeAgents}
        window.location.href = contactUrl;
    };
    
    const handleUpgrade = async (type) => {
        const agentCount = universalAgentCount;
        const totalCost = agentCount * type.price;
        const updateType = type.subscription_type_id > parseInt(subscriptionType) ? 'Upgrade' : 'Downgrade';    
        const perAgentCost = type.price; // Make sure this is the correct calculation
        
        if(type.subscription_type_id === 3){
            window.location.href = '/contact';
            return;
        }

        if (type.subscription_type_id === parseInt(subscriptionType) && agentCount === parseInt(activeAgents)) {
            //setMessage('Your subscription is already up-to-date with the selected plan and agent count.');
            setMessageType('info');
            //return;
        } 
        if(activeAgents > agentCount ){
            setMessage(`Unable to downgrade to fewer agents than your currently active ${activeAgents} agents. Please adjust the agent count or choose a different plan.`);
            setMessageType('error');
            return;
        }
        try {
            const payload = {
                subscription_id: subscriptionId,
                stripe_subscription_id: stripeSubscriptionId,
                subscription_type_id: type.subscription_type_id,
                subscription_agents_count: agentCount,
                peragent_cost: parseFloat(perAgentCost),
                total_cost: parseFloat(totalCost),
            };

            const response = await axios.post('/saas/update/', payload );        
            const subscriptionData = response.data.subscription;
            const checkoutURL = response.data.checkout_url;
                        
            // Update local storage with new subscription details
            localStorage.setItem('subscription_id', subscriptionData.subscription_id);
            localStorage.setItem('subscription_type', subscriptionData.subscription_type_id);
            localStorage.setItem('subscription_agents', subscriptionData.subscription_agents_count);
            localStorage.setItem('active_agents', subscriptionData.active_agents_count);
            localStorage.setItem('subscription_status', subscriptionData.status);
            // Update state as well
            setSubscriptionType(subscriptionData.subscription_type_id);
            setSubscriptionAgents(subscriptionData.subscription_agents_count);
            setActiveAgents(subscriptionData.active_agents_count);
            setSubscriptionStatus(subscriptionData.status);

            if(!stripeSubscriptionId && checkoutURL){
                window.location.href = checkoutURL;
            }else{
                setMessage('Your subscription has been successfully updated. Changes will reflect in the billing from the next billing cycle. Thank you for continuing to trust our services!');
                setMessageType('success');    
            }
    
        } catch (error) {
            console.error('Error updating subscription:', error);
            setMessage(`Error updating subscription: ${error.message}`);
            setMessageType('error');
        }
    };    
    const handleCancel = async () => {
        if (!subscriptionId) {
            setMessage('No active subscription to cancel.');
            setMessageType('info');
            return;
        }
        try {
            await axios.post(`/saas/cancel_subscription/${subscriptionId}`);
    
            // Clear local storage or update state
            localStorage.removeItem('subscription_id');
            // ...
    
            setMessage('Subscription cancelled successfully.');
            setMessageType('success');
        } catch (error) {
            console.error('Error cancelling subscription:', error);
            setMessage(`Error cancelling subscription: ${error.message}`);
            setMessageType('error');
        }
    };
            
    return (
        <div className="website-container">
            <center>
                <div className='content-container'>
                {!isBillingPage  && (
                    <section className="main-section">
                        <h1>Flexible Pricing for Every Need</h1>
                        <p>Choose the plan that best fits your business, with features tailored to every scale.</p>
                    </section>
                )}
                {message && <div className={`message ${messageType}`}>{message}</div>}
                    <section className="pricing-section">
                        <table className="pricing-table">
                            <thead>
                                <tr>
                                    {subscriptionTypes.map(type => (
                                        <th key={type.subscription_type_id} className={type.subscription_type_id === parseInt(subscriptionType) ? 'selected-plan' : ''}>
                                            <div>
                                            <center>  
                                                <h2><center>{type.name}</center> </h2>
                                                <p>{type.target}</p>                                                
                                            </center>     
                                            </div>
                                        </th>
                                    ))}
                                </tr>
                                <tr>
                                {subscriptionTypes.map(type => (
                                        <th key={type.subscription_type_id} className={type.subscription_type_id === parseInt(subscriptionType) ? 'selected-plan' : ''}>

                                <center>
                                                {subscriptionId ? (
                                                    type.subscription_type_id === parseInt(subscriptionType) ? (
                                                        <div>
                                                        <span className='active-plan'>This is your current plan. Its status is {subscriptionStatus.charAt(0).toUpperCase() + subscriptionStatus.slice(1)}</span> <br/>
                                                        <button onClick={() => handleUpgrade(type)}>
                                                            Update
                                                        </button>
                                                        </div>
                                                    ) : (
                                                        <button onClick={() => handleUpgrade(type)}>
                                                            {
                                                                // Determine if the action is an upgrade, downgrade or requires contact
                                                                type.subscription_type_id > parseInt(subscriptionType) ? 'Upgrade to this Plan' :
                                                                type.subscription_type_id < parseInt(subscriptionType) ? 'Downgrade to this Plan' :
                                                                'Contact Us for Upgrade'
                                                            }
                                                        </button>
                                                    )
                                                ) : (
                                                    <button onClick={() => handleSubscribe(type)}>Subscribe</button>
                                                )}
                                            </center>
                                            </th>
                                    ))}

                                </tr>
                            </thead>
                            <tbody>
                            <tr>
                                <td colSpan={4}>
                                <center>
                                    <button onClick={() => setUniversalAgentCount(prevCount => Math.max(prevCount - 1, activeAgents, 1))} className='pricingbtn'>-</button>
                                    <span className='counttxt'>{universalAgentCount}</span>
                                    <button onClick={() => setUniversalAgentCount(prevCount => prevCount + 1)} className='pricingbtn'>+</button>
                                    <p>Customer Experience AI Agents</p>
                                </center>

                                </td>
                                </tr>                                
                                <tr>
                                    {subscriptionTypes.map(type => (
                                        <td key={type.subscription_type_id} className={type.subscription_type_id === parseInt(subscriptionType) ? '' : ''}>
                                            <center className='pricingtxt'> 
                                            {renderPrice(type)}
                                            </center> 
                                        </td>
                                    ))}
                                </tr>

                                <tr>
                                    {subscriptionTypes.map(type => (
                                        <td key={type.subscription_type_id} className={type.subscription_type_id === parseInt(subscriptionType) ? '' : ''}>
                                            <center>
                                                {type.responses_limit === -1 ? 'Custom Responses' :
                                                type.responses_limit ? `${type.responses_limit} Responses/ Agent/ month` : 'Unlimited Responses'}
                                            </center> 
                                        </td>
                                    ))}
                                </tr>
                                <tr>
                                    {subscriptionTypes.map(type => (
                                        <td key={type.subscription_type_id} className={type.subscription_type_id === parseInt(subscriptionType) ? '' : ''}>
                                            {renderFeatures(type.features)}
                                        </td>
                                    ))}
                                </tr>
                                <tr>
                                    {subscriptionTypes.map(type => (
                                        <td key={type.subscription_type_id} className={type.subscription_type_id === parseInt(subscriptionType) ? '' : ''}>
                                            <center>
                                            {subscriptionId ? (
                                                    type.subscription_type_id === parseInt(subscriptionType) ? (
                                                        <p className='active-plan'>Active Subscription Plan</p>
                                                    ) : (
                                                        <button onClick={() => handleUpgrade(type)}>
                                                            {type.subscription_type_id === 3 ? 'Contact Us' : 'Upgrade'}
                                                        </button>
                                                    )
                                                ) : (
                                                    <button onClick={() => handleSubscribe(type)}>Subscribe</button>
                                                )}
                                            </center>
                                        </td>
                                    ))}
                                </tr>
                            </tbody>
                        </table>
                    </section>

                </div>
            </center>
        </div>
    );
};

export default WebPricing;
