

import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Link, useLocation } from 'react-router-dom';
import '../../themes/blue/Billing.css';
import { Helmet } from 'react-helmet';

const WebPricing = () => {
    const [subscriptionTypes, setSubscriptionTypes] = useState([]);
    const [universalAgentCount, setUniversalAgentCount] = useState(() => {
        const activeAgents = parseInt(localStorage.getItem('active_agents'), 10);
        return activeAgents > 0 ? activeAgents : 1;
    });
    const userId = localStorage.getItem('user_id');
    const accountId = localStorage.getItem('account_id');
    const [message, setMessage] = useState('');
    const [messageType, setMessageType] = useState('');
    const location = useLocation();
    const isBillingPage = location.pathname.startsWith('/billing');
    const [showFeatures, setShowFeatures] = React.useState({});

    const subscriptionId = localStorage.getItem('subscription_id') || '';
    const [subscriptionType, setSubscriptionType] = useState(localStorage.getItem('subscription_type') || -1);
    const [activeAgents, setActiveAgents] = useState(parseInt(localStorage.getItem('active_agents') || 0));
    const [subscriptionAgents, setSubscriptionAgents] = useState(parseInt(localStorage.getItem('subscription_agents') || 0));
    const [subscriptionStatus, setSubscriptionStatus] = useState(localStorage.getItem('subscription_status') || "");
    const [stripeSubscriptionId, setStripeSubscriptionId] = useState(localStorage.getItem('stripe_subscription_id') || null);

    useEffect(() => {
        const fetchSubscriptionTypes = async () => {
            try {
                const response = await axios.get('/saas/subscription-types/');
                setSubscriptionTypes(response.data);

                const storedPlanId = localStorage.getItem('planId');
                const storedAgentCount = localStorage.getItem('agentCount');
                if (storedPlanId && storedAgentCount) {
                    setSubscriptionType(storedPlanId);
                    setUniversalAgentCount(parseInt(storedAgentCount));
                    localStorage.removeItem('planId');
                    localStorage.removeItem('agentCount');
                }
            } catch (error) {
                console.error('There was an error fetching the subscription types', error);
            }
        };
        fetchSubscriptionTypes();
    }, []);

    useEffect(() => {
        const fetchSubscriptionDetails = async () => {
            if (subscriptionId) {
                try {
                    const response = await axios.get(`/saas/subscription/${subscriptionId}`);
                    const subscriptionData = response.data;
                    setSubscriptionType(subscriptionData.subscription_type_id);
                    setSubscriptionAgents(subscriptionData.subscription_agents_count);
                    setUniversalAgentCount(subscriptionData.subscription_agents_count);
                    setActiveAgents(subscriptionData.active_agents_count);
                    setSubscriptionStatus(subscriptionData.status);
                    setStripeSubscriptionId(subscriptionData.stripe_subscription_id);
                    localStorage.setItem('subscription_status', subscriptionData.status);
                } catch (error) {
                    console.error('Error fetching subscription details:', error);
                }
            }
        };
        fetchSubscriptionDetails();
    }, [subscriptionId]);

    const handleSubscribe = async (type) => {
        const agentCount = universalAgentCount;
        const totalCost = agentCount * type.price;
        const perAgentCost = type.price;

        if (!userId) {
            localStorage.setItem('referenceURL', '/billing');
            localStorage.setItem('planId', type.subscription_type_id.toString());
            localStorage.setItem('agentCount', agentCount.toString());
            window.location.href = `/signup`;
            return;
        }

        try {
            const payload = {
                user_id: userId,
                account_id: accountId,
                subscription_type_id: type.subscription_type_id,
                subscription_agents_count: agentCount,
                active_agents_count: activeAgents,
                peragent_cost: parseFloat(perAgentCost),
                total_cost: parseFloat(totalCost),
            };

            // 🔍 Log the request payload
            console.log("📤 Sending payload to /saas/subscribe:", payload);

            const response = await axios.post('/saas/subscribe', payload);

            // ✅ Log the response
            console.log("✅ Response from /saas/subscribe:", response.data);

            const subscriptionData = response.data.subscription;
            const checkoutURL = response.data.checkout_url;

            setMessage('New Subscription has been created successfully.');
            setMessageType('success');

            localStorage.setItem('subscription_id', subscriptionData.subscription_id);
            localStorage.setItem('subscription_type', subscriptionData.subscription_type_id);
            localStorage.setItem('subscription_agents', subscriptionData.subscription_agents_count);
            localStorage.setItem('active_agents', subscriptionData.active_agents_count);
            localStorage.setItem('subscription_status', subscriptionData.status);

            setSubscriptionType(subscriptionData.subscription_type_id);
            setSubscriptionAgents(subscriptionData.subscription_agents_count);
            setActiveAgents(subscriptionData.active_agents_count);
            setSubscriptionStatus(subscriptionData.status);

            if (checkoutURL) {
                window.location.href = checkoutURL;
            }
        } catch (error) {
            console.error('❌ Error subscribing:', error);
            setMessage('Error subscribing: ' + error.message);
            setMessageType('error');
        }
    };

    const toggleFeatures = (id) => {
        const featuresList = document.getElementById(`features-list-${id}`);
        if (featuresList.classList.contains('show')) {
            featuresList.classList.remove('show');
        } else {
            featuresList.classList.add('show');
        }
    };

    const renderPrice = (type) => {
        const total = universalAgentCount * type.price;
        if (type.price === -1) return <span>Contact for pricing</span>;
        return total === 0 ? <span>$0 - It's Free</span> : <span>${total} / total / month </span>;
    };

    const renderFeatures = (features) => {
        return (
            <ul>
                {features.map((feature, index) => (
                    <li key={index}>
                        <span className="material-symbols-outlined">check</span>{feature}
                    </li>
                ))}
            </ul>
        );
    };

    return (
        <>
            <Helmet>
                <title>Pricing - Thought: Unlocking Affordable Customer Excellence</title>
                <meta name="description" content="Explore our flexible pricing plans at Thought - designed to empower businesses of all sizes with Generative AI-driven customer engagement solutions. Find the perfect fit for your business and elevate your customer experience today." />
            </Helmet>

            <div className={!isBillingPage ? 'website-container' : ''}>
                <div className="pricing-container">
                    <center>
                        {!isBillingPage && (
                            <section className="main-section" style={{ boxShadow: 'none', paddingBottom: '0px', marginBottom: '20px' }}>
                                <h1>Flexible Pricing for every need</h1>
                                <Link to='/signup' className="homebutton">Signup for free</Link>
                                <p>Choose the plan that best fits your business, with features tailored to every scale. You can signup for free and start building your agent.</p>
                            </section>
                        )}
                        {message && <div className={`message ${messageType}`}>{message}</div>}

                        <p>Please select the number of AI Customer Experience agents you wish to deploy. Pricing per agent will be based on the selected plan. You can build up to two agents for free, but to publish them on your website, you must subscribe.</p>
                        <br />
                        <div className="agent-count-selector">
                            <button onClick={() => setUniversalAgentCount(prev => Math.max(prev - 1, activeAgents, 1))}>-</button>
                            <span>{universalAgentCount}</span>
                            <button onClick={() => setUniversalAgentCount(prev => prev + 1)}>+</button>
                            {universalAgentCount === 1 ? 'AI Agent' : 'AI Agents'}
                        </div>

                        <div className="pricing-plans">
                            {subscriptionTypes.map((type) => (
                                <div key={type.subscription_type_id} className={`plan ${type.subscription_type_id === parseInt(subscriptionType) ? 'selected-plan' : ''}`}>
                                    <h2>{type.name}</h2>
                                    <p>{type.target}</p>
                                    <h2 className='plan-pricing'>
                                        {type.price === -1 || type.price === null || type.price === 0 ? 'Custom' : `$${type.price}`}
                                    </h2>
                                    <p>per agent / month</p>
                                    <strong className='plan-pricing-total'>{renderPrice(type)}</strong>
                                    <p>{type.responses_limit === -1 ? 'Custom Responses' : type.responses_limit ? `${type.responses_limit} Responses / Agent` : 'Unlimited Responses'}</p>

                                    <div className="plan-actions">
                                        {type.price === -1 ? (
                                            <p><Link to='/contact' className='buttonstyle'>Contact Us</Link></p>
                                        ) : (
                                            subscriptionId && subscriptionStatus === 'active' ? (
                                                type.subscription_type_id === parseInt(subscriptionType) ? (
                                                    <>
                                                        <button onClick={() => handleSubscribe(type)}>Update</button><br /><br />
                                                        <span className='active-plan'>This is your current plan. Its status is {subscriptionStatus.charAt(0).toUpperCase() + subscriptionStatus.slice(1)}</span>
                                                    </>
                                                ) : (
                                                    <>
                                                        <button onClick={() => handleSubscribe(type)}>
                                                            {type.subscription_type_id > parseInt(subscriptionType) ? 'Upgrade' :
                                                                type.subscription_type_id < parseInt(subscriptionType) ? 'Change' :
                                                                    'Contact Us for Upgrade'}
                                                        </button><br /><br />
                                                        <span className='active-plan'>&nbsp;</span>
                                                    </>
                                                )
                                            ) : (
                                                <button onClick={() => handleSubscribe(type)}>Get Started</button>
                                            )
                                        )}
                                    </div>

                                    <a href={`#features-list-${type.subscription_type_id}`} className="toggle-features-btn" onClick={() => toggleFeatures(type.subscription_type_id)}>
                                        <span className="material-symbols-outlined">expand_circle_down</span>View Features
                                    </a>
                                    <div id={`features-list-${type.subscription_type_id}`} className="features-list">
                                        {renderFeatures(type.features)}
                                    </div>
                                </div>
                            ))}
                        </div>
                    </center>
                </div>
            </div>
        </>
    );
};

export default WebPricing;

