import React, { useEffect, useRef, useState } from 'react';
import { GoogleMap, Marker, useLoadScript } from '@react-google-maps/api';
import { BackgroundFetch } from '@transistorsoft/capacitor-background-fetch';

const TEST_LOCATIONS = [
  { lat: 11.0168, lng: 76.9558 }, // Central Coimbatore (Town Hall)
  { lat: 11.0340, lng: 76.9600 }, // Gandhipuram
  { lat: 11.0170, lng: 77.0000 }, // Peelamedu
  { lat: 11.0510, lng: 76.9830 }, // Saibaba Colony
  { lat: 10.9970, lng: 76.9660 }, // Singanallur
  { lat: 11.0300, lng: 76.9340 }, // RS Puram
  { lat: 11.0085, lng: 76.9769 }, // Race Course
  { lat: 11.0730, lng: 77.0180 }, // Saravanampatti
  { lat: 10.9946, lng: 77.0027 }, // Ondipudur
  { lat: 11.0527, lng: 76.9249 }, // Vadavalli
];



const libraries = ['places'];
const mapContainerStyle = { width: '100%', height: '400px' };

function VehicleTracking() {
  const [isConnected, setIsConnected] = useState(false);
  const [currentPosition, setCurrentPosition] = useState(TEST_LOCATIONS[0]);
  const [backendMessages, setBackendMessages] = useState([]);
  const [trackingData, setTrackingData] = useState([]);
  const [lastUpdate, setLastUpdate] = useState(null);
  const [isSimulating, setIsSimulating] = useState(false);
  const [simulationSpeed, setSimulationSpeed] = useState(3000);

  const simulationRef = useRef(null);
  const wsRef = useRef(null);

  const { isLoaded, loadError } = useLoadScript({
    googleMapsApiKey: 'AIzaSyC_eJxdZDuzciiczsYYohGvvb108g2_9os',
    libraries,
  });

  // Send location via WS (foreground only)
  const sendLocationWS = (location) => {
    if (wsRef.current && isConnected) {
      const data = {
        device_id: 'dev123',
        location: {
          type: 'Point',
          coordinates: [location.lng, location.lat],
        },
        speed: 45.5,
        heading: 180,
        fuel: 80,
        UUID: 'UUUFDSUDSUU254454587',
        timestamp: new Date().toISOString(),
      };
      wsRef.current.send(JSON.stringify(data));
      console.log('Sent WS:', data);
    }
  };

  // Send location via HTTP fetch (background fetch)
  const sendLocationHTTP = async (location) => {
    try {
      const payload = {
        device_id: 'Taxi A',
        location: {
          type: 'Point',
          coordinates: [location.lng, location.lat],
        },
        speed: 45.5,
        heading: 180,
        fuel: 80,
        UUID: 'UUUFDSUDSUU254454587',
        timestamp: new Date().toISOString(),
      };

      const response = await fetch('http://localhost:8003/api/tracking_data', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload),
      });
      if (!response.ok) throw new Error('Network response was not ok');
      console.log('Sent HTTP:', payload);
    } catch (error) {
      console.error('Error sending HTTP location:', error);
    }
  };

  const toggleSimulation = () => {
    if (isSimulating) {
      clearInterval(simulationRef.current);
      setBackendMessages((prev) => [...prev, 'Simulation stopped']);
    } else {
      let index = 0;
      sendLocationWS(TEST_LOCATIONS[index]);
      simulationRef.current = setInterval(() => {
        index = (index + 1) % TEST_LOCATIONS.length;
        const location = TEST_LOCATIONS[index];
        setCurrentPosition(location);
        sendLocationWS(location);
        setBackendMessages((prev) => [...prev, `Sent Location ${index + 1}`]);
      }, simulationSpeed);
      setBackendMessages((prev) => [...prev, 'Simulation started']);
    }
    setIsSimulating(!isSimulating);
  };

  const initWebSocket = () => {
    const ws = new WebSocket('ws://localhost:8003/ws/tracking_data?token=67fec97f95809e100c9954d4');
    wsRef.current = ws;

    ws.onopen = () => {
      setIsConnected(true);
      setBackendMessages((prev) => [...prev, 'Connected to WebSocket']);
    };

    ws.onmessage = (event) => {
      try {
        const data = JSON.parse(event.data);
        if (data.message) {
          setBackendMessages((prev) => [...prev.slice(-9), data.message]);
        }
        if (data.status === 'ok' && data.data) {
          setTrackingData((prev) => [...prev.slice(-4), data.data]);
          setLastUpdate(new Date().toLocaleTimeString());
        }
      } catch (error) {
        setBackendMessages((prev) => [...prev, 'Error parsing message']);
      }
    };

    ws.onerror = (e) => {
      console.error('WebSocket error event:', e);
      setBackendMessages((prev) => [...prev, `WebSocket error occurred`]);
    };

    ws.onclose = () => {
      setIsConnected(false);
      setBackendMessages((prev) => [...prev, 'WebSocket closed']);
    };
  };

  const setupBackgroundFetch = async () => {
    try {
      const status = await BackgroundFetch.configure(
        {
          minimumFetchInterval: 15, // iOS minimum is ~15 min
          stopOnTerminate: false,
          enableHeadless: true,
          startOnBoot: true,
          requiredNetworkType: BackgroundFetch.NETWORK_TYPE_ANY,
        },
        async (taskId) => {
          console.log('[BackgroundFetch] Task:', taskId);
          try {
            // IMPORTANT: You cannot use WebSocket here because background fetch runs in native headless mode
            // Send location using HTTP request in background
            // Here, for demo, use the currentPosition from the last known location in React state
            // In real app, persist latest location in local storage or native storage to access here

            await sendLocationHTTP(currentPosition);

            console.log('[BackgroundFetch] Location sent via HTTP.');
          } catch (error) {
            console.error('Background WebSocket send error:', error);
          }
          BackgroundFetch.finish(taskId);
        },
        (error) => {
          console.error('[BackgroundFetch] Failed to start:', error);
        }
      );

      console.log('[BackgroundFetch] configured. Status:', status);
    } catch (err) {
      console.error('BackgroundFetch init error:', err);
    }
  };

  useEffect(() => {
    initWebSocket();
    setupBackgroundFetch();

    return () => {
      wsRef.current?.close();
      clearInterval(simulationRef.current);
    };
  }, []);

  if (loadError) return <div>Error loading maps</div>;
  if (!isLoaded) return <div>Loading maps...</div>;

  return (
    <div style={{ padding: '20px' }}>
      <h2>Vehicle Tracking Dashboard</h2>

      <button onClick={toggleSimulation}>
        {isSimulating ? 'Stop' : 'Start'} Simulation
      </button>

      <select
        disabled={isSimulating}
        value={simulationSpeed}
        onChange={(e) => setSimulationSpeed(Number(e.target.value))}
      >
        <option value={1000}>Fast (1s)</option>
        <option value={3000}>Normal (3s)</option>
        <option value={5000}>Slow (5s)</option>
      </select>

      <p>Status: {isConnected ? '🟢 Connected' : '🔴 Disconnected'}</p>
      <p>Last Update: {lastUpdate || 'N/A'}</p>

      <GoogleMap mapContainerStyle={mapContainerStyle} center={currentPosition} zoom={12}>
        <Marker position={currentPosition} />
      </GoogleMap>

      <h3>Backend Messages</h3>
      <div style={{ height: 150, overflowY: 'auto', backgroundColor: '#eee', padding: '10px' }}>
        {backendMessages.map((msg, i) => (
          <div key={i}>{msg}</div>
        ))}
      </div>

      <h3>Tracking Data</h3>
      <div style={{ height: 150, overflowY: 'auto', backgroundColor: '#eee', padding: '10px' }}>
        {trackingData.map((data, i) => (
          <div key={i}>
            <p>
              <strong>Device:</strong> {data.device_id} <br />
              <strong>Lat:</strong> {data.location?.coordinates?.[1]?.toFixed(6)} <br />
              <strong>Lng:</strong> {data.location?.coordinates?.[0]?.toFixed(6)} <br />
              <strong>Time:</strong> {new Date(data.timestamp).toLocaleTimeString()}
            </p>
          </div>
        ))}
      </div>
    </div>
  );
}

export default VehicleTracking;
