import React, { useEffect, useState } from 'react';
import axios from 'axios';

const TaskBar = ({ customerId, workforceId, vehicleId }) => {
  const [summary, setSummary] = useState(null);
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchTaskSummary = async () => {
      try {
        const params = new URLSearchParams();
        if (customerId) params.append('customer_id', customerId);
        if (workforceId) params.append('workforce_id', workforceId);
        if (vehicleId) params.append('vehicle_id', vehicleId);

        const response = await axios.get(`/tasks/summary/?${params.toString()}`);
        setSummary(response.data);
      } catch (err) {
        setError('Failed to load task stats');
      } finally {
        setLoading(false);
      }
    };

    fetchTaskSummary();
  }, [customerId, workforceId, vehicleId]);

  if (loading) return <div style={{ padding: '10px', textAlign: 'center' }}>Loading task stats...</div>;
  if (error) return <div style={{ color: 'red', padding: '10px' }}>{error}</div>;
  if (!summary) return null;

  return (
    <div style={{
      margin: '1rem 0',
      padding: '15px',
      borderRadius: '8px',
      backgroundColor: '#f8f9fa',
      border: '1px solid #e9ecef'
    }}>
      <h4 style={{ marginBottom: '15px', color: '#495057' }}>Task Statistics</h4>
      <div style={{
        display: 'flex',
        flexWrap: 'wrap',
        gap: '20px',
        alignItems: 'flex-start'
      }}>
        {/* Total Tasks */}
        <div style={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          gap: '5px',
          padding: '12px',
          backgroundColor: 'white',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.05)',
          minWidth: '80px'
        }}>
          <span className="material-symbols-outlined" style={{ fontSize: '24px', color: 'black' }}>assessment</span>
          <span style={{ fontSize: '12px', color: '#666', textAlign: 'center' }}>Total Tasks</span>
          <span style={{ fontSize: '18px', fontWeight: '600' }}>{summary.total_tasks || 0}</span>
        </div>

        {/* Active Tasks */}
        <div style={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          gap: '5px',
          padding: '12px',
          backgroundColor: 'white',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.05)',
          minWidth: '80px'
        }}>
          <span className="material-symbols-outlined" style={{ fontSize: '24px', color: 'black' }}>autorenew</span>
          <span style={{ fontSize: '12px', color: '#666', textAlign: 'center' }}>Active</span>
          <span style={{ fontSize: '18px', fontWeight: '600' }}>{summary.active_tasks || 0}</span>
        </div>

        {/* Completed Tasks */}
        <div style={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          gap: '5px',
          padding: '12px',
          backgroundColor: 'white',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.05)',
          minWidth: '80px'
        }}>
          <span className="material-symbols-outlined" style={{ fontSize: '24px', color: 'black' }}>check_circle</span>
          <span style={{ fontSize: '12px', color: '#666', textAlign: 'center' }}>Completed</span>
          <span style={{ fontSize: '18px', fontWeight: '600' }}>{summary.completed_tasks || 0}</span>
        </div>

        {/* Cancelled Tasks */}
        <div style={{
          display: 'flex',
          flexDirection: 'column',
          alignItems: 'center',
          gap: '5px',
          padding: '12px',
          backgroundColor: 'white',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.05)',
          minWidth: '80px'
        }}>
          <span className="material-symbols-outlined" style={{ fontSize: '24px', color: 'black' }}>cancel</span>
          <span style={{ fontSize: '12px', color: '#666', textAlign: 'center' }}>Cancelled</span>
          <span style={{ fontSize: '18px', fontWeight: '600' }}>{summary.cancelled_tasks || 0}</span>
        </div>
      </div>
    </div>
  );
};

export default TaskBar;
