
import React, { useState, useEffect, useCallback, useRef } from 'react';
import axios from 'axios';
import '../../themes/blue/SearchPage.css';

const SearchPage = ({
  rowsPerPage = 10,
  fetchDataUrl,
  itemKeyMapping,
  ListComponent,
  actionFormats = [],
  additionalParams = {},
  identifierKey,
  onActionClick,
  dataKey = 'items',
  refreshKey,
  searchFields = [],
  onSearch,
  onPageChange,
  currentPage = 0,
  onCellClick,
}) => {
  const defaultTextField = [{ name: 'q', type: 'text', placeholder: 'Search...', defaultValue: '' }];
  const effectiveFields = searchFields.length > 0 ? searchFields : defaultTextField;

  // Initialize searchFilters state properly
  const [searchFilters, setSearchFilters] = useState(() => {
    const init = {};
    effectiveFields.forEach(f => {
      init[f.name] = f.defaultValue || '';
    });
    return init;
  });

  // Added originalItems state to store the unfiltered list
  const [originalItems, setOriginalItems] = useState([]);
  const [items, setItems] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [page, setPage] = useState(currentPage);
  const [totalCount, setCount] = useState(0);
  const [activePopup, setActivePopup] = useState(null);
  const [sortBy, setSortBy] = useState('');
  const [sortOrder, setSortOrder] = useState('asc');
  const [selectedRow, setSelectedRow] = useState(null);
  const [isSearchActive, setIsSearchActive] = useState(false); // Track if search is active

  const totalPages = Math.ceil(totalCount / rowsPerPage);
  const memoizedActionClick = useCallback(onActionClick, []);
  const initialRender = useRef(true);

  const startItem = page * rowsPerPage + 1;
  const endItem = Math.min((page + 1) * rowsPerPage, totalCount);

  const handleFilterChange = (fieldName, value) => {
    setSearchFilters(prev => ({ ...prev, [fieldName]: value }));
  };

  const buildQueryParams = () => {
    const params = new URLSearchParams({
      ...additionalParams,
      skip: page * rowsPerPage,
      limit: rowsPerPage,
      ...searchFilters,
    });

    if (sortBy) {
      params.set('sort_by', sortBy);
      params.set('sort_order', sortOrder);
    }

    window.history.replaceState(null, '', `?${params.toString()}`);
    return Object.fromEntries(params);
  };

  const fetchItems = async () => {
    setIsLoading(true);
    try {
      const response = await axios.get(
        typeof fetchDataUrl === 'function' ? fetchDataUrl(buildQueryParams()) : fetchDataUrl,
        { params: typeof fetchDataUrl === 'function' ? {} : buildQueryParams() }
      );

      const fetchedItems = Array.isArray(response.data[dataKey]) ? response.data[dataKey] : [];
      
      // Store both original and filtered items
      setOriginalItems(fetchedItems);
      setItems(fetchedItems);
      setCount(response.data.total_count || 0);
      setSelectedRow(null);
      setIsSearchActive(false); // Reset search active state on new fetch
      if (onSearch) onSearch(searchFilters);
    } catch (e) {
      console.error('Fetch error:', e);
    } finally {
      setIsLoading(false);
    }
  };

  const resetAndFetch = () => {
    setPage(0);
    setCount(0);
    fetchItems();
  };

  // Modified handleSearchSubmit to filter items based on search term
  const handleSearchSubmit = (e) => {
    e.preventDefault();
    const searchTerm = searchFilters.q?.toLowerCase().trim();
    
    if (searchTerm) {
      // Filter items based on search term
      const filtered = originalItems.filter(item => {
        // Check if any of the item's values include the search term
        return Object.values(item).some(
          value => value && value.toString().toLowerCase().includes(searchTerm)
        );
      });
      setItems(filtered);
      setCount(filtered.length);
      setIsSearchActive(true);
    } else {
      // If search term is empty, show all items
      setItems(originalItems);
      setCount(originalItems.length);
      setIsSearchActive(false);
    }
  };

  const clearFilters = () => {
    // Reload the entire page
    window.location.reload();
  };

  const handleRowClick = (item, index) => {
    setSelectedRow(index === selectedRow ? null : index);
  };

  const handlePageChange = (newPage) => {
    if (newPage >= 0 && newPage < totalPages) {
      setPage(newPage);
      if (onPageChange) onPageChange(newPage);
    }
  };

  const handleSort = (columnKey) => {
    if (!columnKey || columnKey === 'actions') return;

    if (sortBy === columnKey) {
      setSortOrder(prev => (prev === 'asc' ? 'desc' : 'asc'));
    } else {
      setSortBy(columnKey);
      setSortOrder('asc');
    }
  };

  const renderSortIcon = (columnKey) => {
    if (!columnKey || columnKey === 'actions') return null;
    if (sortBy !== columnKey) return <span className="sort-icon">↕</span>;
    return sortOrder === 'asc' ? <span className="sort-icon">↑</span> : <span className="sort-icon">↓</span>;
  };

  const renderPageNumbers = () => {
    const pages = [];
    const maxVisiblePages = 5;
    let startPage = Math.max(0, page - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages - 1, startPage + maxVisiblePages - 1);

    if (endPage - startPage + 1 < maxVisiblePages) {
      startPage = Math.max(0, endPage - maxVisiblePages + 1);
    }

    if (page > 0) {
      pages.push(
        <span key="first" className="pagination-page" onClick={() => handlePageChange(0)}>⏮ First</span>
      );
    }

    for (let i = startPage; i <= endPage; i++) {
      pages.push(
        <span
          key={i}
          onClick={() => handlePageChange(i)}
          className={`pagination-page ${page === i ? 'active' : ''}`}
        >
          {i + 1}
        </span>
      );
    }

    if (page < totalPages - 1) {
      pages.push(
        <span key="last" className="pagination-page" onClick={() => handlePageChange(totalPages - 1)}>Last ⏭</span>
      );
    }

    return pages;
  };

  useEffect(() => {
    if (!initialRender.current) fetchItems();
    else {
      initialRender.current = false;
      fetchItems();
    }
  }, [page, sortBy, sortOrder]);

  useEffect(() => {
    if (!initialRender.current) resetAndFetch();
  }, [refreshKey]);

  const activeFilters = effectiveFields.filter(f => searchFilters[f.name]);

  return (
    <div className="projects-list-container">
      {/* Search Input */}
      <form onSubmit={handleSearchSubmit} className="search-bar-row">
        <input
          type="text"
          placeholder="Search..."
          value={searchFilters.q || ''}
          onChange={(e) => handleFilterChange('q', e.target.value)}
          className="search-box"
        />
      </form>

      {/* Filter bar */}
      <div className="filter-bar">
        <div className="filter-left">
          {effectiveFields.filter(f => f.name !== 'q').map(field => (
            <span
              key={field.name}
              className="filter_button"
              onClick={() => setActivePopup(activePopup === field.name ? null : field.name)}
            >
              + {field.placeholder || field.name}
            </span>
          ))}
          {(searchFilters.q || isSearchActive || activeFilters.length > 0) && (
            <span 
              style={{ fontWeight: 'bold', cursor: 'pointer' }} 
              onClick={clearFilters}
            >
              Clear Filters
            </span>
          )}
        </div>
        <div className="filter-right">
          <button className="export-button">
            <span className="material-symbols-outlined">upload</span> Export
          </button>
        </div>
      </div>

      {/* Filter popup */}
      {activePopup && (
        <div className="filter_popup">
          <div className="filter_popup_header">
            <strong>Filter by {effectiveFields.find(f => f.name === activePopup)?.placeholder || activePopup}</strong>
            <a href='#' className="filter_popup_close" onClick={() => setActivePopup(null)}>×</a>
          </div>
          {(() => {
            const field = effectiveFields.find(f => f.name === activePopup);
            if (!field) return null;
            if (field.type === 'dropdown') {
              return (
                <select
                  value={searchFilters[field.name]}
                  onChange={(e) => handleFilterChange(field.name, e.target.value)}
                >
                  {field.options.map(opt => (
                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                  ))}
                </select>
              );
            } else {
              return (
                <input
                  type={field.type}
                  placeholder={field.placeholder}
                  value={searchFilters[field.name]}
                  onChange={(e) => handleFilterChange(field.name, e.target.value)}
                />
              );
            }
          })()}
          <div style={{ display: 'flex', gap: '10px', marginTop: '10px' }}>
            <button className="apply-button" onClick={() => { resetAndFetch(); setActivePopup(null); }}>Apply</button>
          </div>
        </div>
      )}

      {/* List Component */}
      <div className="list-content">
        <ListComponent
          items={items}
          mapping={itemKeyMapping}
          headers={Object.keys(itemKeyMapping)}
          actionFormats={actionFormats}
          identifierKey={identifierKey}
          onActionClick={memoizedActionClick}
          onCellClick={onCellClick} 
          selectedRow={selectedRow}
          onRowClick={handleRowClick}
          sortBy={sortBy}
          sortOrder={sortOrder}
          onSort={handleSort}
          renderSortIcon={renderSortIcon}
        />

        {totalCount > 0 && (
          <div className="pagination-wrapper">
            <div className="pagination-info-text">
              Displayed {startItem}-{endItem} of {totalCount}
            </div>
            <div className="pagination-controls">
              {renderPageNumbers()}
            </div>
          </div>
        )}
      </div>

      {items.length === 0 && !isLoading && <p>No items found.</p>}
    </div>
  );
};

export default SearchPage;