import React, { useState, useEffect, useCallback, useRef } from 'react';
import axios from 'axios';

const SearchPage = ({
  initialSearchQuery = '',
  rowsPerPage = 10,
  fetchDataUrl,
  itemKeyMapping,
  ListComponent,
  actionFormats = [],
  additionalParams = {},
  identifierKey,
  onActionClick,
  dataKey = 'items',
  onTotalCountChange,
  onItemsFetched,


}) => {
  const [items, setItems] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [searchQuery, setSearchQuery] = useState(initialSearchQuery);
  const [page, setPage] = useState(0);
  const [total_count, setCount] = useState(0);
  const [shouldFetchMore, setShouldFetchMore] = useState(true);
  const memoizedActionClick = useCallback(onActionClick, []);
  const initialRender = useRef(true);

  //console.log("Serarch loading [page] ");

  const fetchItems = async () => {
    if (isLoading || (!shouldFetchMore && page > 0)) return; // Prevent loading more when not needed
  
    setIsLoading(true);

    try {
      const response = await axios.get(fetchDataUrl, {
        params: {
          ...additionalParams,
          skip: page * rowsPerPage,
          limit: rowsPerPage,
          q: searchQuery,
        },
      });

      //console.log(response);
      const fetchedItems = Array.isArray(response.data[dataKey])
        ? response.data[dataKey]
        : [];

      if (onItemsFetched && typeof onItemsFetched === 'function') {
        onItemsFetched(fetchedItems);
      }
    
      if (page === 0) {
        setItems(fetchedItems);
      } else {
        setItems((prevItems) => [...prevItems, ...fetchedItems]);
      }

      if ('total_count' in response.data) {
        setCount(response.data.total_count);
      }
      if (onTotalCountChange) {
        onTotalCountChange(total_count); // Assuming total_count is the variable holding the total count of items
      }
            
      setShouldFetchMore(fetchedItems.length === rowsPerPage);
    } catch (error) {
      console.error("Error fetching items:", error);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    if (initialRender.current) {
        initialRender.current = false;
    } else {
        fetchItems();
    }
}, [page]);

useEffect(() => {
    if (!initialRender.current) {
      fetchItems(); // Fetch data on page change, but not on initial render
    }
  }, [page, searchQuery]); // Added searchQuery as a dependency

  const handleLoadMore = () => {
    if (!isLoading && shouldFetchMore) {
      setPage((prevPage) => prevPage + 1);
    }
  };

  const handleSearch = (e) => {
    if (e.key === "Enter" && (searchQuery.length > 3 || searchQuery.length === 0)) {
      setPage(0);  // Reset page number for a new search
      setShouldFetchMore(true); // Ensure we can load more results for the new search
    }
  };

  const handleScrollToTop = () => {
    window.scrollTo(0, 0);
  };

  return (
    <div className='projects-list-container' id={identifierKey}>
      {items.length > 0 || isLoading ? (
        <>
          <div className="search-and-total">
            <input
              type="text"
              placeholder="Search ..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              onKeyDown={handleSearch}
              className="search-box"
            />
            <span className="total-projects">{total_count} items in total</span>
          </div>
          <div className="list-content">
            <ListComponent
              items={items}
              mapping={itemKeyMapping}
              onAddClick={() => { }}
              headers={Object.keys(itemKeyMapping)}
              actionFormats={actionFormats}
              identifierKey={identifierKey}
              onActionClick={memoizedActionClick}
            />
            {shouldFetchMore && (
          <div className='load-more-container'>
          <a onClick={handleLoadMore} disabled={isLoading} className='load-more'>
            Load More
            <span className="material-symbols-outlined">navigate_next</span>
          </a>
          <a href={`#${identifierKey}`}  className="back-to-top">
            ↑ Top
          </a>
          </div>            
        )}
          </div>
          {items.length === 0 && !isLoading && <p>No items found.</p>}
        </>
      ) : (
        <p>There are no items to display.</p>
      )}
    </div>
  );
};

export default SearchPage;
