import React, { useEffect, useState } from 'react';
import axios from 'axios';
import FormBuilder from '../../common/Builder/FormBuilder';

function IntegrationModal({ appId, integrationId, onClose }) {
    const [showSuccess, setShowSuccess] = useState(false);
    const [appDetails, setAppDetails] = useState(null);
    const [integrationDetails, setIntegrationDetails] = useState(null);
    const [dynamicFields, setDynamicFields] = useState([]);
    const baseURL = new URL(axios.defaults.baseURL).origin;
    const [defaultValues, setDefaultValues] = useState({});
    const [schemas, setSchemas] = useState({}); // New state for schemas

    useEffect(() => {
        const fetchAppDetails = async (app_id) => {
            try {
                let response;
                if(app_id)
                    response = await axios.get(`/appflow/${app_id}`);
                else
                    response = await axios.get(`/appflow/${appId}`);

                if (response.status === 200) {
                    setAppDetails(response.data);
                    await loadDynamicFields(response.data);
                }
            } catch (error) {
                console.error('Error fetching app details:', error);
            }
        };

        const fetchIntegrationDetails = async () => {
            try {
                const response = await axios.get(`/appflow/integrations/${integrationId}`);
                if (response.status === 200) {
                    setIntegrationDetails(response.data);
                    fetchAppDetails(response.data.app_id);
                }
            } catch (error) {
                console.error('Error fetching integration details:', error);
            }
        };

        if (appId) {
            fetchAppDetails();
        }

        if (integrationId) {
            fetchIntegrationDetails();
        }
    }, [appId, integrationId]);

    const loadDynamicFields = async (appDetails) => {

        if (appDetails && appDetails.api_schema) {
            let schema;
                // Parse the JSON string to an object
            let parsedSchema = typeof appDetails.api_schema === 'string' ? JSON.parse(appDetails.api_schema) : appDetails.api_schema;
    
            // Check if parsedSchema has a 'url' property
            if (parsedSchema && parsedSchema.url) {
                try {
                    // Fetch the schema from the URL
                    const response = await axios.get("json/" + parsedSchema.url);
                    schema = response.data;
                } catch (error) {
                    console.error('Error fetching schema from URL:', error);
                    return; // Exit if fetching fails
                }
            } else {
                // Use the parsed schema directly
                schema = parsedSchema;
            }
            
            const fields = mapSchemaToFields(schema,appDetails);
            setDynamicFields(fields);

        }
    };
    
    const mapSchemaToFields = (schema,appDetails) => {
        let fields = [];
    
        const accountCredentialsSchema = schema.components.schemas.AccountCredentials;
        const configurationsSchema = schema.components.schemas.Configurations;
        setSchemas({ accountCredentials: schema.components.schemas.AccountCredentials, configurations: schema.components.schemas.Configurations });

        if(appDetails){
            setDefaultValues({
                name: appDetails.name || '',
                auth: appDetails.auth || '',
                // Add other fields as needed
            });
            fields.push({ name: 'name', type: 'text', placeholder: 'Enter the app integration name', mandatory: true, label:'Name your integration', value: appDetails.name });
            fields.push({ name: 'auth', type: 'text', placeholder: 'Enter the app integration name', mandatory: true, label:'Auth Type', value: appDetails.auth });

        }
        // Map AccountCredentials properties to fields
        if (accountCredentialsSchema && accountCredentialsSchema.properties) {
            Object.entries(accountCredentialsSchema.properties).forEach(([key, value]) => {
                fields.push({
                    name: key,
                    type: 'text',
                    placeholder: value.description,
                    label: key,
                    mandatory: accountCredentialsSchema.required.includes(key)
                });
            });
        }
    
        // Map Configurations properties to fields
        if (configurationsSchema && configurationsSchema.properties) {
            Object.entries(configurationsSchema.properties).forEach(([key, value]) => {
                fields.push({
                    name: key,
                    type: value.type === 'string' ? 'text' : (value.type === 'text' ? 'textarea' : 'text'),
                    placeholder: value.description,
                    label: key,
                    mandatory: configurationsSchema.required.includes(key),
                    rows: value.type === 'text' ? 4 : undefined // Assuming text type maps to a textarea
                });
            });
        }
        
        console.log(fields);
        return fields;
    };
        
    const handleFinalSubmission = async (formData) => {
        const userId = localStorage.getItem('user_id');
        const accountId = localStorage.getItem('account_id');

        const url = integrationId ? `/appflow/integrations/view/${integrationId}` : '/appflow/subscribe';
        const method = integrationId ? 'put' : 'post';

        let configurations = {};
        let authConfig = { 
            auth: formData.auth, // Assuming the server expects this field
            credentials: {} 
        };
        const accountCredentialsSchema = schemas.accountCredentials;
        const configurationsSchema = schemas.configurations;

        //const authFields = Object.keys(accountCredentialsSchema.properties);

        // Iterate through formData and populate authConfig and configurations

        Object.keys(formData).forEach(key => {
            if (accountCredentialsSchema.properties && key in accountCredentialsSchema.properties) {
                // If the key is part of authConfig, add it to credentials
                authConfig.credentials[key] = formData[key];
            } 
            if (configurationsSchema.properties && key in configurationsSchema.properties) {
                // Otherwise, add it to configurations
                configurations[key] = formData[key];
            }
        });
        
        // Prepare the final data to be sent to the server
        const integrationData = {
            title: formData.name,
            app_id: appId,
            user_id: userId,
            account_id: accountId,
            configurations: configurations,
            auth_config: authConfig,
            status: 1 // or other status based on your logic
        };

        console.log(integrationData);

        try {
            const response = await axios({ method, url, data: integrationData });
            if (response.status === 200) {
                setShowSuccess(true);
            }
        } catch (error) {
            console.error('Error submitting the form:', error);
        }
    };

    return (
        <div className="modal-overlay">
            <div className="modal-content">
                <h3>{appDetails?.name || 'Integration'} &nbsp;&nbsp;&nbsp;
                    <a href="#" onClick={onClose}><span className="material-symbols-outlined">close</span></a>
                </h3>
                
                {showSuccess ? (
                    <div>
                        <p>Integration {integrationId ? 'updated' : 'added'} successfully!</p>
                        <button onClick={onClose}>Close</button>
                    </div>
                ) : (
                    <FormBuilder 
                        key={integrationId + JSON.stringify(integrationDetails)} 
                        config={{ formType: 'chat', steps: [{ fields: dynamicFields, defaultValues: integrationDetails || defaultValues || {} }] }}
                        onFinalSubmit={handleFinalSubmission} 
                    />
                )}
            </div>
        </div>
    );
}

export default IntegrationModal;

