import React, { useState, useEffect } from 'react';
import axios from 'axios';

const ProjectFlow = ({ flowId, projectId, onClose }) => {
  const [flowDetail, setFlowDetail] = useState({});
  const [context, setContext] = useState('');
  const [welcome, setWelcome] = useState('');
  const [display, setDisplay] = useState('');
  const [isIntegrated, setIsIntegrated] = useState(false);
  const [error, setError] = useState(''); // Add state for error message
  const [isSubmitting, setIsSubmitting] = useState(false);

  useEffect(() => {
    const fetchFlowDetails = async () => {
      try {
        const response = await axios.get(`/flows/view/${flowId}/`);
        setFlowDetail(response.data);
        setContext(response.data.context);
      } catch (error) {
        console.error("Error fetching flow details:", error);
      }
    };
    fetchFlowDetails();
  }, [flowId]);

  const validateFields = () => {
    if (!context.trim() || context.trim().length < 10) {
      setError('Context is required and must be a minimum of 10 characters.');
      return false;
    }
    if (!welcome.trim() || welcome.trim().length < 10) {
      setError('Introduction message is required and must be a minimum of 10 characters.');
      return false;
    }
    setError(''); // Clear error message
    return true;
  };

  const handleIntegrate = async () => {
    if (!validateFields()) {
      return; // Prevent submission if validation fails
    }
    if(isSubmitting){
      return;
    }
    setIsSubmitting(true);
    try {
      const response = await axios.post(`/flows/projectflow/${projectId}/${flowId}/`, {
        context: context,
        welcome: welcome,
        display: display,
      });
      if (response.status === 200 || response.status === 201) {
        setIsIntegrated(true); // Mark the integration as successful
        setError(''); // Clear error message
      }
    } catch (error) {
      setError('Failed to integrate flow. Please try again.');
    }
    setIsSubmitting(false);
  };

  return (
    <div className="modal-overlay">
      <div className="modal-content">
        {isIntegrated ? (
          <div>
            <h2>Flow Integration</h2>
            <h3>Integration Successful!</h3>
            <div className='closeontoprightflow'>
            <a href="#" onClick={onClose} ><span className="material-symbols-outlined">close</span></a>
            </div>
            <p>Flow has been integrated to this AI Agent.</p>

          </div>
        ) : (
          <>
            <h2>Flow Integration</h2>
            <div className='closeontoprightflow'>
            <a href="#" onClick={onClose} ><span className="material-symbols-outlined">close</span></a>
            </div>
            {error && <p className='error'>{error}</p>}
            <div>
              <strong>Flow Name:</strong> {flowDetail.name}
              <br/><br/>
            </div>
            <div>
              <strong>Goal:</strong> {flowDetail.goal}
              <br/><br/>
            </div>
            <div>
              <label>
              Context (for multiple context, Comma Separated)
              </label>
                <br/>
                <input
                  type="text"
                  value={context}
                  onChange={(e) => setContext(e.target.value)}
                />
                <br/>
                <p>This is the when a flow is executed when it semantically matches with user's intent.</p>
            </div>
            <div>
              <label>
                Introduction message (optional)
                <br/>
                <input
                  type="text"
                  value={welcome}
                  placeholder='Introduction message to explain or welcome.'
                  onChange={(e) => setWelcome(e.target.value)}
                />
              </label>
            </div>
            <div>
              <label>
              Display Name of Button
                <br/>
                <input
                  type="text"
                  value={display}
                  placeholder='Display name of Action button.'
                  onChange={(e) => setDisplay(e.target.value)}
                />
              </label>
            </div>
            <button onClick={handleIntegrate} disabled={isSubmitting}>{isSubmitting? 'Integrating...': 'Integrate'}</button>
          </>
        )}
      </div>
    </div>
  );
};

export default ProjectFlow;
