import React, { useState, useEffect } from 'react';
import axios from 'axios';

const ProjectDefaultFlow = ({ appId, projectId, flowId, endPoint, integrationId, onClose }) => {
  const [flowDetail, setFlowDetail] = useState({});
  const [context, setContext] = useState('');
  const [welcome, setWelcome] = useState('');
  const [display, setDisplay] = useState('');
  const [isIntegrated, setIsIntegrated] = useState(false);
  const accountId = localStorage.getItem('account_id');
  const userId = localStorage.getItem('user_id');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [errorMessage, setErrorMessage] = useState(''); // State for submission error


  useEffect(() => {
    const fetchFlowDetails = async () => {
      try {
        const response = await axios.get(`/flows/view/${flowId}/`);
        setFlowDetail(response.data);
        setContext(response.data.context);
        setWelcome(response.data.welcome || ''); // Initialize welcome with the fetched data if available
        setDisplay(response.data.display || ''); // Initialize display with the fetched data if available

      } catch (error) {
        console.error("Error fetching flow details:", error);
      }
    };
    fetchFlowDetails();
  }, [flowId]);

  const handleSubmit = async () => {
    if(isSubmitting) return;
    setErrorMessage('');
    if (!flowDetail.name || flowDetail.name.length < 10) {
      setErrorMessage('Flow name is mandatory and must be at least 10 characters.');
      return;
    }
    if (!flowDetail.goal || flowDetail.goal.length < 10) {
      setErrorMessage('Flow goal is mandatory and must be at least 10 characters.');
      return;
    }
    if (!context || context.length < 6) {
      setErrorMessage('Context is mandatory and must be at least 6 characters.');
      return;
    }
    setIsSubmitting(true);

    const dataToSend = {
      template_flow_id: flowId,
      app_integration_id: integrationId,
      project_id: projectId,
      name: flowDetail.name,
      goal: flowDetail.goal,
      context: context,
      user_id:userId,
      account_id:accountId,
      endpoint: endPoint,
      welcome: welcome,
      display: display,
    };

    try {
      const response = await axios.post(`/flows/predefined/`, dataToSend);
      if (response.status === 200 || response.status === 201) {
        setIsIntegrated(true); // Indicate success
      }
    } catch (error) {
      setErrorMessage('Failed to submit predefined flow. Please try again.');
    }
    setIsSubmitting(false);
  };

  return (
    <div className="modal-overlay">
      <div className="modal-content">
        {isIntegrated ? (
          <div>
            <h2>Flow Integration</h2>
            <h3>Integration Successful!</h3>
            <div className='closeontoprightflow'>
            <a href="#" onClick={onClose} ><span className="material-symbols-outlined">close</span></a>
            </div>
            <p>Flow has been integrated to this AI Agent.</p>
          </div>
        ) : (
          <>        
            <h2>Flow Integration</h2>
            <div className='closeontoprightflow'>
            <a href="#" onClick={onClose} ><span className="material-symbols-outlined">close</span></a>
            </div>
            {errorMessage && <div className="error">{errorMessage}</div>}
            <div>
              <label>Flow Title</label>
              <input
                type="text"
                value={flowDetail.name || ''}
                onChange={(e) => setFlowDetail({ ...flowDetail, name: e.target.value })}
              />
              <br/><br/>
            </div>
            <div>
              <label>Goal</label>
              <input
                type="text"
                value={flowDetail.goal || ''}
                onChange={(e) => setFlowDetail({ ...flowDetail, goal: e.target.value })}
              />
              <br/><br/>
            </div>
            <div>
              <label>
                Context (for multiple context, Comma Separated)
                <input
                  type="text"
                  value={context}
                  onChange={(e) => setContext(e.target.value)}
                />
              </label>
              <br/>
              <p>This is the when a flow is executed when it semantically matches with user's intent.</p>
            </div>
            <div>
              <label>
              Introduction message (optional)
                <input
                  type="text"
                  value={welcome} // Corrected to use welcome
                  placeholder='Introduction message to explain or welcome.'
                  onChange={(e) => setWelcome(e.target.value)}
                />
              </label>
            </div>
            <div>
              <label>
              Display Name of Button
                <input
                  type="text"
                  value={display} // Corrected to use display
                  placeholder='Enter the action button display name!'
                  onChange={(e) => setDisplay(e.target.value)}
                />
              </label>
              
            </div>
            <button onClick={handleSubmit} disabled={isSubmitting}>  {isSubmitting ? 'Submitting...' : 'Submit'}</button>
          </>
        )}
      </div>
    </div>
  );
};

export default ProjectDefaultFlow;
