import React, { useEffect, useState } from 'react';
import axios from 'axios';
import FormBuilder from '../../common/Builder/FormBuilder';

function IntegrationModal({ appId, integrationId, onClose, onIntegrationSuccess }) {
    const [showSuccess, setShowSuccess] = useState(false);
    const [appDetails, setAppDetails] = useState(null);
    const [integrationDetails, setIntegrationDetails] = useState(null);
    const [dynamicFields, setDynamicFields] = useState([]);
    const baseURL = new URL(axios.defaults.baseURL).origin;
    const [defaultValues, setDefaultValues] = useState({});
    const [schemas, setSchemas] = useState({}); // New state for schemas
    const [integration_app_id, setAppId] = useState(null);

    useEffect(() => {
        // Function to fetch app details
        const fetchAppDetails = async (app_id) => {
            try {
                const response = await axios.get(`/appflow/${app_id || appId}`);
                if (response.status === 200) {
                    setAppDetails(response.data);
                    await loadDynamicFields(response.data);
                }
            } catch (error) {
                console.error('Error fetching app details:', error);
            }
        };

        // Function to fetch integration details and then app details
        const fetchIntegrationDetails = async () => {
            try {
                const response = await axios.get(`/appflow/integrations/${integrationId}`);
                //console.log(response);
                if (response.status === 200) {
                    setIntegrationDetails(response.data);
                    fetchAppDetails(response.data.app_id);
                    setAppId(response.data.app_id);
                    //console.log(integrationDetails);
                }
            } catch (error) {
                console.error('Error fetching integration details:', error);
            }
        };

        if (appId) {
            fetchAppDetails();
        }

        if (integrationId) {
            fetchIntegrationDetails();
        }
    }, [appId, integrationId]);

  
useEffect(() => {
    if (integrationDetails && appDetails) {
        loadDynamicFields(appDetails, integrationDetails);
    }
}, [integrationDetails, appDetails]);
  
    const loadDynamicFields = async (appDetails, integrationDetails) => {
        if (appDetails && appDetails.api_schema) {
            let schema;
            let parsedSchema = typeof appDetails.api_schema === 'string' ? JSON.parse(appDetails.api_schema) : appDetails.api_schema;

            if (parsedSchema && parsedSchema.url) {
                try {
                    const response = await axios.get("json/"+parsedSchema.url);
                    schema = response.data;
                } catch (error) {
                    console.error('Error fetching schema from URL:', error);
                    return;
                }
            } else {
                schema = parsedSchema;
            }
            
            setSchemas({
                //accountCredentials: schema.components.schemas.AccountCredentials,
                //configurations: schema.components.schemas.Configurations
                accountCredentials: schema.AccountCredentials,
                configurations: schema.Configurations
            });
            const fields = mapSchemaToFields(schema, appDetails);
            setDynamicFields(fields);
        }
    };

    const mapSchemaToFields = (schema, appDetails) => {
        let fields = [];
        //const accountCredentialsSchema = schema.components.schemas.AccountCredentials;
        //const configurationsSchema = schema.components.schemas.Configurations;
        const accountCredentialsSchema = schema.AccountCredentials;
        const configurationsSchema = schema.Configurations;

        if(appDetails){
            if(!integrationDetails){setDefaultValues({title: appDetails.name || '',auth: appDetails.auth || ''});}
            fields.push({ name: 'title', type: 'text', placeholder: 'Enter the app integration name', mandatory: true, label:'Name your integration', value: integrationDetails?.title || appDetails.name || '' });
            fields.push({ name: 'auth', type: 'hidden', placeholder: 'Type of Auth', mandatory: true, value:  integrationDetails?.auth_config.auth || appDetails.auth || '' });
        }

        if (accountCredentialsSchema && accountCredentialsSchema.properties) {
            Object.entries(accountCredentialsSchema.properties).forEach(([key, value]) => {
                fields.push({
                    name: key,
                    type: 'text',
                    placeholder: value.description,
                    label: key,
                    mandatory: accountCredentialsSchema.required.includes(key),
                    value: integrationDetails?.auth_config?.credentials[key] || ''
                });
            });
        }
    
        if (configurationsSchema && configurationsSchema.properties) {
            Object.entries(configurationsSchema.properties).forEach(([key, value]) => {
                fields.push({
                    name: key,
                    type: value.type === 'string' ? 'text' : (value.type === 'text' ? 'textarea' : 'text'),
                    placeholder: value.description,
                    label: key,
                    mandatory: configurationsSchema.required.includes(key),
                    value: integrationDetails?.configurations[key] || '',
                    rows: value.type === 'text' ? 4 : undefined
                });
            });
        }

        return fields;
    };

    const handleFinalSubmission = async (formData) => {
        const userId = localStorage.getItem('user_id');
        const accountId = localStorage.getItem('account_id');
        const url = integrationId ? `/appflow/subscribe/${integrationId}` : '/appflow/subscribe';
        const method = integrationId ? 'put' : 'post';

        let configurations = { ...integrationDetails?.configurations }; // Start with existing configurations
        let authConfig = { auth: formData.auth || integrationDetails?.auth_config.auth, credentials: {} };
        let authConfigChanged = false;
        
        Object.keys(formData).forEach(key => {
            if (schemas.accountCredentials?.properties && key in schemas.accountCredentials.properties) {

                if (formData[key] !== "**********") {
                    authConfig.credentials[key] = formData[key];
                    authConfigChanged = true;
                }else{
                    authConfigChanged = false;
                }
                    
            } else if (schemas.configurations?.properties && key in schemas.configurations.properties) {
                configurations[key] = formData[key];
            }
        });

        const integrationData = {
            title: formData.title || integrationDetails?.title,
            app_id: integration_app_id || appId,
            user_id: userId,
            account_id: accountId,
            configurations,
            auth_config: authConfig,
            status: 1
        };

        // Only include auth_config in the payload if it was changed
        if (authConfigChanged) {
            integrationData.auth_config = authConfig;
        }

        try {
            const response = await axios({ method, url, data: integrationData });
            if (response.status === 200) {
                setShowSuccess(true);
                if(onIntegrationSuccess){
                    onIntegrationSuccess(response.data.integration_id); // Pass the integrationId to the parent
                }

            }
        } catch (error) {
            console.error('Error submitting the form:', error);
        }
    };

    return (
        <div className="modal-overlay">
            <div className="modal-content">
                <h2>{appDetails?.name || 'Integration'} &nbsp;&nbsp;&nbsp;
                </h2>
                <div className='closeontoprightflow'>
                <a href="#" onClick={onClose} ><span className="material-symbols-outlined">close</span></a>
                </div>
                
                {showSuccess ? (
                    <div>
                        <p>Integration {integrationId ? 'updated' : 'added'} successfully!</p>
                    </div>
                ) : (
                    <>
                    <FormBuilder 
                        key={integrationId + JSON.stringify(integrationDetails)} 
                        config={{ formType: 'chat', steps: [{ fields: dynamicFields, defaultValues: integrationDetails || defaultValues }] }}
                        onFinalSubmit={handleFinalSubmission} 
                    />
                    { appDetails && !appDetails.status && (
                    <div className="app-status">
                        <p className='error'>This app integration is in testing mode. It might not work as expected.</p>
                    </div>
                    )}

                    </>
                )}
            </div>
        </div>
    );
}

export default IntegrationModal;
