import React, { useEffect, useState } from 'react';
import axios from 'axios';

function ProjectModal({ projectId, onClose }) {
  const [project, setProject] = useState(null);
  const [content, setContent] = useState(null);
  const [stat, setStat] = useState(null);

  useEffect(() => {
    async function fetchProject() {
      try {
        const response = await axios.get(`/projects/v/detail/${projectId}/`);
        setProject(response.data);
      } catch (error) {
        console.error("Failed to fetch project:", error);
      }
    }

    async function fetchContent() {
        try {
          const response = await axios.get(`/projects/v/all-content/${projectId}/`);
          setContent(response.data[0]);
          //console.log("Content Data:", response);

        } catch (error) {
          console.error("Failed to fetch content:", error);
        }
      }

      async function fetchStats() {
        try {
          const response = await axios.get(`/projects/v/project-statistics/${projectId}/`);
          setStat(response.data);
          //console.log("Stat Data:", response);

        } catch (error) {
          console.error("Failed to fetch content:", error);
        }
      }
      
      if (projectId) {
        fetchProject();
        fetchContent();
        fetchStats();
      }
          
  }, [projectId]);

  if (!project) return null;

  // Helper function to convert the timestamp to a readable format
  function formatDate(timestamp) {
    const dateObj = new Date(timestamp);
    return dateObj.toLocaleDateString("en-US", {
      year: 'numeric', month: 'long', day: 'numeric',
      hour: '2-digit', minute: '2-digit', second: '2-digit',
      hour12: true
    });
  }

  function ensureArray(item) {
    if (Array.isArray(item)) return item;
    if (item) return [item];
    return [];
  }
   
  return (
    <div className="modal-overlay">
      <div className="modal-content">
        <h2>{project.title} <a href="#" onClick={onClose}><span className="material-symbols-outlined">close</span></a></h2>
        
        <div className="project-detail-section">
          <p>{project.description}</p>
          <p><strong>Goal:</strong> {project.goal}</p>
          <p><strong>Agent Name:</strong> {project.cx_agent_name}</p>
          <p><strong></strong> {formatDate(project.updated_date)}</p>
          <p>{project.status}</p>
        </div>

        {stat && (
        <div className="statistics-row">
                <span className="statistics-value"><strong>{stat.interaction_count}</strong></span>
                <span className="statistics-label">Interactions</span>
                <span className="statistics-value"><strong>{stat.visitor_count}</strong></span>
                <span className="statistics-label">Visitors</span>
                <span className="statistics-value"><strong>{stat.goal_count}</strong></span>
                <span className="statistics-label">Goals</span>
        </div>
        )}
        <div className="content-detail-section">
        <h3>Data Sources</h3>
        {content && (
            <>
                {content.main_url && ensureArray(content.main_url).length > 0 && (
                    <p>
                        {ensureArray(content.main_url).map((url, index) => (
                            <React.Fragment key={index}>
                                <a href={url} target="_blank" rel="noopener noreferrer">{url}</a>
                                {index !== content.main_url.length - 1 && <br />}
                            </React.Fragment>
                        ))}
                    </p>
                )}

                {content.youtube_urls && ensureArray(content.youtube_urls).length > 0 && (
                    <p>YouTube Videos <br/> 
                        {ensureArray(content.youtube_urls).map((url, index) => (
                            <React.Fragment key={index}>
                                <a href={url} target="_blank" rel="noopener noreferrer">{url}</a>
                                {index !== content.youtube_urls.length - 1 && <br />}
                            </React.Fragment>
                        ))}
                    </p>
                )}

                {content.google_drive_link && ensureArray(content.google_drive_link).length > 0 && (
                    <p>Google Drive <br/> 
                        {ensureArray(content.google_drive_link).map((link, index) => (
                            <React.Fragment key={index}>
                                <a href={link} target="_blank" rel="noopener noreferrer">{link}</a>
                                {index !== content.google_drive_link.length - 1 && <br />}
                            </React.Fragment>
                        ))}
                    </p>
                )}

                {content.dropbox_link && ensureArray(content.dropbox_link).length > 0 && (
                    <p>Dropbox <br/>
                        {ensureArray(content.dropbox_link).map((link, index) => (
                            <React.Fragment key={index}>
                                <a href={link} target="_blank" rel="noopener noreferrer">{link}</a>
                                {index !== content.dropbox_link.length - 1 && <br />}
                            </React.Fragment>
                        ))}
                    </p>
                )}

                {content.file_paths && ensureArray(content.file_paths).length > 0 && (
                    <p>Files <br/>
                        {ensureArray(content.file_paths).map((file, index) => (
                            <React.Fragment key={index}>
                                {file.filename || file}
                                {index !== content.file_paths.length - 1 && <br />}
                            </React.Fragment>
                        ))}
                    </p>
                )}
            </>
        )}

        {/* ... your other code ... */}

        <button >Add more data</button>

        </div>

      </div>
    </div>
  );
}

export default ProjectModal;
