import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import { useSelector } from 'react-redux';
import '../../themes/blue/Menu.css';


const Menu = () => {
  const location = useLocation();
  const roleRights = useSelector((state) => state.rbac.roleRights);
  const roleFeatures = useSelector((state) => state.rbac.roleFeatures);
  const [menuItems, setMenuItems] = useState([]);
  const [collapsed, setCollapsed] = useState(false);


  useEffect(() => {
    const features = roleFeatures?.filter(feature => feature.display === true) || [];


    const combinedMenu = features.map(feature => {
      const featureKey = String(feature.features_id || feature.id);
      const permissions = roleRights?.[featureKey] || [];
      return {
        ...feature,
        key: featureKey,
        path: feature.url,
        permissions,
      };
    });


    const accessibleMenu = combinedMenu.filter(item => item.permissions.includes('read'));
    setMenuItems(accessibleMenu);
  }, [roleRights, roleFeatures]);


  const normalizePath = (path) => path.replace(/\/$/, '');
  const isMenuItemActive = (path) =>
    normalizePath(location.pathname) === normalizePath(path);


  const renderUserDetails = () => {
    const name = localStorage.getItem('name');
    const profileImageUrl = localStorage.getItem('picurl');
    return (
      <div className={`profile-row ${collapsed ? 'collapsed' : ''}`}>
        {profileImageUrl ? (
          <img src={profileImageUrl} alt="Profile" className="profile-image-menu" />
        ) : (
          <span className="profile-placeholder-menu">
            {name ? name.charAt(0).toUpperCase() : ''}
          </span>
        )}
        {!collapsed && (
          <span className="user-name-menu">
            <a href="/settings">{name}</a>
          </span>
        )}
      </div>
    );
  };


  return (
    <aside className={`menu ${collapsed ? 'collapsed' : ''}`}>
      <div className="main-menu">
        <div className="menu-header">
          <a href="/" className="logo-text">
            <img
              src="/logo_dark.png"
              alt="Logo"
              className={`menu-logo ${collapsed ? 'collapsed' : ''}`}
            />
          </a>
          <span className="menu-toggle" onClick={() => setCollapsed(!collapsed)}>
            <span className="material-symbols-outlined">
              {collapsed ? 'chevron_right' : 'chevron_left'}
            </span>
          </span>
        </div>


        {menuItems.map(item => {
          const itemName = item.name?.trim().toLowerCase();


          return (
            <div
              key={item.key}
              className={`menu-item-wrapper ${isMenuItemActive(item.path) ? 'selected' : ''}`}
            >
              <Link
                to={collapsed ? '#' : item.path}
                className={`menu-item ${isMenuItemActive(item.path) ? 'selected' : ''}`}
                onClick={(e) => {
                  if (collapsed) {
                    e.preventDefault();
                    setCollapsed(false);
                  } else if (itemName === 'track') {
                    setCollapsed(true);
                  }
                }}
              >
                {item.icon && (
                  <span className="material-symbols-outlined">{item.icon}</span>
                )}
                {!collapsed && <span className="menu-text">{item.name}</span>}
              </Link>
            </div>
          );
        })}
      </div>


      <div className="secondary-menu">
        {renderUserDetails()}
      </div>
    </aside>
  );
};


export default Menu;
