
import React, { useState } from 'react';
import formatDate from '../../../utils/date';
import { Link } from 'react-router-dom';
import '../../themes/blue/SearchPage.css';

// ✅ Extracted hoverable clickable cell
const HoverableCell = ({ header, value, item, mappingValue, onCellClick }) => {
  const [isHovered, setIsHovered] = useState(false);
  const cellContent = value ? value.toString().charAt(0).toUpperCase() + value.toString().slice(1) : 'None';

  return (
    <button
      className="clickable-cell"
      style={{
        background: 'none',
        border: 'none',
        padding: 0,
        cursor: 'pointer',
        font: 'inherit',
        textAlign: 'left',
        color: isHovered ? '#0066cc' : 'black',
        textDecoration: isHovered ? 'underline' : 'none'
      }}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
      onClick={(e) => {
        e.stopPropagation();
        console.log('✅ Cell clicked:', header);
        console.log('📦 Item:', item);
        console.log('🧩 Mapping:', mappingValue);
        if (typeof onCellClick === 'function') {
          onCellClick(header, item);
        } else {
          console.warn('⚠️ onCellClick not defined or not a function');
        }
      }}
    >
      {cellContent}
    </button>
  );
};

const List = ({
  items = [],
  headers = [],
  mapping = {},
  truncateLength = 45,
  actionFormats = [],
  onActionClick,
  onCellClick,
  identifierKey = 'id',
  dataKey = 'items',
  sortBy,
  sortOrder,
  onSort,
  renderSortIcon,
}) => {
  const [expandedRow, setExpandedRow] = useState(-1);

  const handleActionClick = (e, idValue, format) => {
    e.preventDefault();
    e.stopPropagation();
    if (typeof onActionClick === 'function') {
      onActionClick(idValue, format);
    }
  };

  const truncateText = (text) => {
    if (typeof text !== 'string') text = String(text);
    return text.length > truncateLength ? text.substring(0, truncateLength) + '...' : text;
  };

  const renderActionLinks = (idValue) => {
    const item = items.find(item => item[identifierKey] === idValue);
    if (!item) return null;

    return actionFormats
      .filter(format => !format.showIf || format.showIf(item))
      .map((format, idx) => {
        const url = format.url?.replace("{id}", idValue) || '#';
        return format.modal ? (
          <a
            key={`${format.label}-${idx}`}
            href="#"
            onClick={(e) => handleActionClick(e, idValue, format)}
            className="actionable"
          >
            {format.label}
          </a>
        ) : (
          <Link key={format.label + '-' + idx} to={url} className="actionable">
            {format.label}
          </Link>
        );
      });
  };

  const capitalizeFirstLetter = (value) => {
    if (typeof value !== 'string') value = String(value);
    return value.charAt(0).toUpperCase() + value.slice(1).toLowerCase();
  };

  const getStatusStyle = (status) => {
    const statusMap = {
      new: "status-new",
      processing: "status-processing",
      failed: "status-failed",
      active: "status-active",
      pending: "status-pending",
      completed: "status-completed",
      inactive: "status-inactive",
      true: "status-active",
      false: "status-inactive",
      1: "status-active",
      2: "status-inactive"
    };
    return statusMap[String(status).toLowerCase()] || '';
  };

  const renderCellContent = (header, value, item) => {
    const mappingValue = mapping[header];
    const isClickable = mappingValue?.clickable === true;
    const cellContent = capitalizeFirstLetter(truncateText(value));

    if (header === "Status") {
      return (
        <span className={`tag ${getStatusStyle(value)}`}>
          {capitalizeFirstLetter(value)}
        </span>
      );
    }

    if ((header.includes("Date") || header.includes(" Date")) && value !== 'N/A') {
      return formatDate(value);
    }

    if (isClickable) {
      return (
        <HoverableCell
          header={header}
          value={cellContent}
          item={item}
          mappingValue={mappingValue}
          onCellClick={onCellClick}
        />
      );
    }

    return cellContent;
  };

  return (
    <table className="list-container">
      <thead>
        <tr className="list-header">
          {headers.map((header) => {
            const mappingValue = mapping[header];
            const key = mappingValue?.key || header;
            const isSortable = mappingValue?.sortable !== false;
            const isActionColumn = key === 'actions';

            return (
              <th
                key={header}
                onClick={() => !isActionColumn && isSortable && onSort?.(key)}
                className={`${isActionColumn ? 'no-sort' : isSortable ? 'sortable-header' : ''}`}
              >
                <div className="header-content">
                  {mappingValue?.label || header}
                  {!isActionColumn && isSortable && renderSortIcon?.(key)}
                </div>
              </th>
            );
          })}
        </tr>
      </thead>
      <tbody>
        {items.map((item, index) => (
          <React.Fragment key={`${item[identifierKey]}-${index}`}>
            <tr
              className={`list-item ${expandedRow === index ? 'selected-row' : ''}`}
              onClick={() => setExpandedRow(index)}
            >
              {headers.map((header) => {
                const mappingValue = mapping[header];
                const key = mappingValue?.key || header;
                const value = item[key] ?? 'N/A';

                return (
                  <td key={`${header}-${value}-${index}`}>
                    {renderCellContent(header, value, item)}
                  </td>
                );
              })}
            </tr>

            {expandedRow === index && (
              <tr key={`actions-${item[identifierKey] || index}`} className="action-row">
                <td colSpan={headers.length}>
                  {renderActionLinks(item[identifierKey])}
                </td>
              </tr>
            )}
          </React.Fragment>
        ))}
      </tbody>
    </table>
  );
};

export default List;
