import axios from 'axios';
import React, { useState, useEffect } from 'react';

const Invite = ({
  onClose,
  defaultRoleId = '',
  defaultAccountId = '',
}) => {
  const [emailList, setEmailList] = useState('');
  const currentUserRoleStr = localStorage.getItem('roles') || "0";
  const currentUserRole = parseInt(currentUserRoleStr, 10);
  const [role, setRole] = useState(defaultRoleId);
  const [roles, setRoles] = useState([]);
  const [isSuccess, setSuccess] = useState(false);
  const [successMessage, setSuccessMessage] = useState('');
  const [errorMessage, setErrorMessage] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Fetch all roles then filter based on the invitation context.
  useEffect(() => {
    const fetchRoles = async () => {
      try {
        const [res, defaultRes] = await Promise.all([
          axios.get('/roles/'),
          axios.get('/roles/d/'),
        ]);
        const allRoles = [
          ...(res.data.roles || []),
          ...(defaultRes.data.roles || []),
        ];
        let filteredRoles = [];

        console.log(currentUserRole);
        if (currentUserRole === 1) {
          // SaaS Super Admin (role 1): can invite all roles with role_id up to 99.
          filteredRoles = allRoles.filter((r) => r.role_id <= 100);
        } else if (currentUserRole === 2) {
          // SaaS User (role 2): can invite roles from 2 to 99.
          filteredRoles = allRoles.filter((r) => r.role_id >= 2 && r.role_id <= 99);
        } else if (currentUserRole === 3 || currentUserRole === 4) {
          // Partner Invitations: allowed roles are 3, 4, and 100.
          filteredRoles = allRoles.filter(
            (r) => r.role_id === 3 || r.role_id === 4 || r.role_id === 100
          );
        } else if (currentUserRole >= 100) {
          // Platform Company (Account Owner): allowed roles are 100 and above.
          filteredRoles = allRoles.filter((r) => r.role_id >= 100);
        }
          

        setRoles(filteredRoles);
        if (!defaultRoleId && filteredRoles.length > 0) {
          setRole(filteredRoles[0].role_id);
        }
      } catch (err) {
        console.error('Failed to fetch roles:', err);
      }
    };
    fetchRoles();
  }, [defaultRoleId, currentUserRole]);

  const handleEmailListChange = (e) => setEmailList(e.target.value);
  const handleRoleChange = (e) => setRole(e.target.value);

  const handleSubmit = (e) => {
    e.preventDefault();
    if (isSubmitting) return;

    setIsSubmitting(true);
    setSuccess(false);
    setSuccessMessage('');
    setErrorMessage('');

    const parsedRole = parseInt(role, 10);
    if (isNaN(parsedRole)) {
      setErrorMessage('Invalid role selected.');
      setIsSubmitting(false);
      return;
    }

    // Build the invitation payload.
    const payload = {
      emails: String(emailList || '').trim(),
      action: 'invite',
      role: parsedRole,
    };

    // Determine which identifier to send based on the invitation type.
    // For Platform Company Invitations (currentUserRole >= 100), always include account_id.
    if (currentUserRole > 100) {
        payload.account_id = defaultAccountId || localStorage.getItem('account_id') || null;
    } else {
      // For SaaS or Partner Invitations:
      // When inviting a partner super admin (3) or for a new account (100),
      // use referral_id and omit account_id.
      if (parsedRole === 3 || parsedRole === 100) {
        payload.referral_id = defaultAccountId || localStorage.getItem('account_id') || null;
        payload.account_id = null;
      } else {
        // This will be for limiting access to the accounts and linking.. 
        payload.account_id =
          defaultAccountId || localStorage.getItem('account_id') || null;
      }
    }

    axios
      .post('/users/invite/', payload)
      .then((res) => {
        if (res.data && res.status === 200) {
          setSuccess(true);
          setSuccessMessage('Invitations sent successfully!');
          setEmailList('');
        } else {
          throw new Error('Failed to send invitations');
        }
      })
      .catch(() => {
        setErrorMessage('Failed to send invitations. Please try again.');
      })
      .finally(() => setIsSubmitting(false));
  };

  return (
    <div>
      <div>
        <h2>Send Invitation</h2>
        {isSuccess ? (
          <p className="success">{successMessage}</p>
        ) : (
          <>
            {errorMessage && <p className="error">{errorMessage}</p>}
            <form onSubmit={handleSubmit}>
              <label htmlFor="emailList">Email Addresses</label>
              <textarea
                id="emailList"
                value={emailList}
                onChange={handleEmailListChange}
                placeholder="Enter emails separated by comma or space"
                disabled={isSubmitting}
                rows={4}
              />
              <label htmlFor="role">Select Role</label>
              <select
                id="role"
                value={role}
                onChange={handleRoleChange}
                disabled={isSubmitting}
              >
                {roles.map((r, index) => (
                  <option key={`role-${r.role_id || index}`} value={r.role_id}>
                    {r.name}
                  </option>
                ))}
              </select>
              <p></p>
              <button type="submit" disabled={isSubmitting}>
                {isSubmitting ? 'Inviting in progress...' : 'Send Invites'}
              </button>
            </form>
          </>
        )}
      </div>
    </div>
  );
};

export default Invite;
