import React, { useState, useEffect } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';

// --- Custom Widgets ---

// AsyncSelectWidget: Loads dropdown options asynchronously.
const AsyncSelectWidget = (props) => {
  const { value, onChange } = props;
  const [loading, setLoading] = useState(true);
  const [options, setOptions] = useState([]);

  useEffect(() => {
    // Simulate an API call with a delay.
    setTimeout(() => {
      setOptions([
        { label: 'USA', value: 'USA' },
        { label: 'Canada', value: 'Canada' },
        { label: 'Mexico', value: 'Mexico' }
      ]);
      setLoading(false);
    }, 1000);
  }, []);

  if (loading) return <div>Loading options...</div>;
  return (
    <select value={value || ''} onChange={e => onChange(e.target.value)}>
      <option value="">Select a country</option>
      {options.map((opt, idx) => (
        <option key={idx} value={opt.value}>{opt.label}</option>
      ))}
    </select>
  );
};

// CustomFlightNumberWidget: Validates flight number via an API simulation on blur.
const CustomFlightNumberWidget = (props) => {
  const { value, onChange, onBlur } = props;
  const [error, setError] = useState('');

  const handleBlur = async (e) => {
    const inputVal = e.target.value;
    // Simulate an API validation call: flight numbers must start with "FL"
    if (inputVal && !inputVal.startsWith('FL')) {
      setError('Flight number must start with "FL"');
    } else {
      setError('');
    }
    if (onBlur) onBlur(e);
  };

  return (
    <div>
      <input
        type="text"
        value={value || ''}
        onChange={(e) => onChange(e.target.value)}
        onBlur={handleBlur}
        aria-invalid={error ? "true" : "false"}
      />
      {error && <span style={{color: 'red'}}>{error}</span>}
    </div>
  );
};

// GoogleAutocompleteWidget: Simulates Google Autocomplete functionality.
const GoogleAutocompleteWidget = (props) => {
  const { value, onChange, onBlur } = props;
  // In a real implementation, integrate Google Places API here.
  return (
    <div>
      <input
        type="text"
        placeholder="Enter location..."
        value={value || ''}
        onChange={(e) => onChange(e.target.value)}
        onBlur={onBlur}
      />
    </div>
  );
};

// GoogleAirportAutocompleteWidget: Simulates a Google Autocomplete specifically for Airports.
const GoogleAirportAutocompleteWidget = (props) => {
  const { value, onChange, onBlur } = props;
  // In a real implementation, integrate the Google Places API restricted to airports.
  return (
    <div>
      <input
        type="text"
        placeholder="Enter airport..."
        value={value || ''}
        onChange={(e) => onChange(e.target.value)}
        onBlur={onBlur}
      />
    </div>
  );
};

// --- JSON Schema ---
const schema = {
  title: "Super Form",
  description: "A form demonstrating advanced features including edit mode, Google Autocomplete, and Airport search.",
  type: "object",
  required: ["firstName", "lastName", "flightNumber", "tripType"],
  properties: {
    firstName: { type: "string", title: "First Name", default: "John" },
    lastName: { type: "string", title: "Last Name" },
    age: { type: "number", title: "Age" },
    flightNumber: { type: "string", title: "Flight Number" },
    tripType: {
      type: "string",
      title: "Trip Type",
      enum: ["one-way", "return"],
      default: "one-way"
    },
    departureDate: {
      type: "string",
      format: "date",
      title: "Departure Date"
    },
    returnDate: {
      type: "string",
      format: "date",
      title: "Return Date"
    },
    destination: {
      type: "string",
      title: "Destination Country"
    },
    // Google Autocomplete for a general location.
    location: {
      type: "string",
      title: "Location (Google Autocomplete)"
    },
    // New property for Airport using Google Autocomplete.
    airport: {
      type: "string",
      title: "Airport"
    },
    upload: {
      type: "string",
      title: "Upload Document",
      format: "data-url",
      description: "File upload (max size 2MB)"
    },
    addresses: {
      type: "array",
      title: "Additional Addresses",
      items: {
        type: "object",
        properties: {
          street: { type: "string", title: "Street" },
          city: { type: "string", title: "City" }
        }
      }
    }
  },
  dependencies: {
    tripType: {
      oneOf: [
        {
          properties: {
            tripType: { enum: ["one-way"] }
          }
        },
        {
          properties: {
            tripType: { enum: ["return"] },
            returnDate: { type: "string", format: "date", title: "Return Date" }
          },
          required: ["returnDate"]
        }
      ]
    }
  }
};

// --- UI Schema ---
const uiSchema = {
  flightNumber: {
    "ui:widget": "customFlightNumber"
  },
  destination: {
    "ui:widget": "asyncSelect"
  },
  location: {
    "ui:widget": "googleAutocomplete"
  },
  airport: {
    "ui:widget": "googleAirportAutocomplete"
  },
  upload: {
    "ui:widget": "file"
  },
  addresses: {
    items: {
      street: { "ui:placeholder": "Enter street" },
      city: { "ui:placeholder": "Enter city" }
    }
  }
};

// --- Custom Validation Function ---
const validate = (formData, errors) => {
  if (formData.age && formData.age < 18) {
    errors.age.addError("Age must be at least 18.");
  }
  return errors;
};

// --- SuperForm Component ---
// Accepts an optional initialData prop for editing and an onDataChange callback.
const SuperForm = ({ initialData, onDataChange }) => {
  const [formData, setFormData] = useState(initialData || {});

  // Auto-save and notify parent: persist form data and send updates via callback.
  useEffect(() => {
    localStorage.setItem("superFormData", JSON.stringify(formData));
    if (onDataChange) onDataChange(formData);
  }, [formData, onDataChange]);

  const onSubmit = ({ formData }) => {
    localStorage.setItem("submittedData", JSON.stringify(formData));
    alert("Form submitted successfully!");
    if (onDataChange) onDataChange(formData);
  };

  // Map custom widgets.
  const widgets = {
    asyncSelect: AsyncSelectWidget,
    customFlightNumber: CustomFlightNumberWidget,
    googleAutocomplete: GoogleAutocompleteWidget,
    googleAirportAutocomplete: GoogleAirportAutocompleteWidget
  };

  return (
    <div style={{ maxWidth: "100%", margin: "0 auto" }}>
      <Form
        schema={schema}
        uiSchema={uiSchema}
        formData={formData}
        validator={validator}
        widgets={widgets}
        onChange={({ formData }) => {
          setFormData(formData);
          console.log("Form changed:", formData);
        }}
        onSubmit={onSubmit}
        validate={validate}
        liveValidate
      />
    </div>
  );
};

export default SuperForm;
