import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { withTheme } from '@rjsf/core';
import MaterialUITheme from '@rjsf/material-ui';
import validator from '@rjsf/validator-ajv8';

const Form = withTheme(MaterialUITheme);

const GenericForm = ({
  collectionName,
  schema,
  uiSchema,
  itemId,
  method = 'POST',
  onSuccess,
  submitUrl,
  initialData = {},
  hideOnSuccess = false,
}) => {
  const [dynamicSchema, setDynamicSchema] = useState(schema);
  const [formData, setFormData] = useState(initialData);
  const [statusMessage, setStatusMessage] = useState('');
  const [showForm, setShowForm] = useState(true);
  const [errors, setErrors] = useState([]);
  const [loading, setLoading] = useState(!!itemId);

  useEffect(() => {
    const loadAsyncEnums = async () => {
      const newSchema = { ...schema, properties: { ...schema.properties } };
      for (const [fieldKey, uiFieldOptions] of Object.entries(uiSchema)) {
        if (uiFieldOptions['ui:options']?.asyncEnum && uiFieldOptions['ui:options']?.fetchUrl) {
          try {
            const res = await axios.get(uiFieldOptions['ui:options'].fetchUrl);
            const items = res.data.items || [];
            newSchema.properties[fieldKey] = {
              ...newSchema.properties[fieldKey],
              enum: items.map(item => item.id),
              enumNames: items.map(item => item.name || item.title),
            };
          } catch (err) {
            console.error(`Error fetching options for ${fieldKey}:`, err);
          }
        }
      }
      setDynamicSchema(newSchema);
    };

    const fetchData = async () => {
      if (itemId) {
        try {
          const res = await axios.get(`/crud/${collectionName}/${itemId}`);
          setFormData(res.data);
        } catch (err) {
          setStatusMessage(`Error fetching ${collectionName} data.`);
        }
        setLoading(false);
      }
    };

    loadAsyncEnums();
    fetchData();
  }, [schema, uiSchema, collectionName, itemId]);

  const checkForDuplicate = async (value, fieldName = 'name') => {
    try {
      const response = await axios.get(
        `/crud/${collectionName}/?q=${encodeURIComponent(value)}`
      );

      return response.data.items.some(item => {
        const fieldValue = item[fieldName] || '';
        const isSameValue = fieldValue.toLowerCase() === value.toLowerCase();

        // Use id or fallback to {collectionName}_id
        const itemIdentifier = item.id || item[`${collectionName}_id`];
        const isDifferentItem = !itemId || itemIdentifier !== itemId;

        return isSameValue && isDifferentItem;
      });
    } catch (error) {
      console.error('Error checking for duplicates:', error);
      return false;
    }
  };

  const handleSubmit = async ({ formData }) => {
    setErrors([]);
    setStatusMessage('');

    const duplicateCheckConfig = {
      checklist: { field: 'name', message: 'A checklist with this name already exists.' },
      pricing_types: { field: 'pricing_type', message: 'A pricing type with this name already exists.' },
      skills: { field: 'name', message: 'A skill with this name already exists.' },
      addons: { field: 'name', message: 'An addon with this name already exists.' }
    };

    if (duplicateCheckConfig[collectionName]) {
      const { field, message } = duplicateCheckConfig[collectionName];
      const valueToCheck = formData[field];

      if (valueToCheck) {
        const isDuplicate = await checkForDuplicate(valueToCheck, field);
        if (isDuplicate) {
          window.alert(message); // Changed to window.alert
          return;
        }
      }
    }

    try {
      const res = await axios({
        method,
        url: submitUrl || (itemId ? `/crud/${collectionName}/${itemId}` : `/crud/${collectionName}/`),
        data: formData,
      });

      if (hideOnSuccess) setShowForm(false);
      if (onSuccess) onSuccess(res.data);
    } catch (err) {
      if (err.response && err.response.status === 409) {
        const message = duplicateCheckConfig[collectionName]?.message ||
          `This ${collectionName} already exists.`;
        window.alert(message); // Changed to window.alert
      } else {
        setStatusMessage(`Error ${itemId ? 'updating' : 'creating'} ${collectionName}.`);
      }
    }
  };

  const handleError = (errors) => {
    setErrors(errors.map(err => err.stack));
  };

  const handleResetForm = () => {
    setShowForm(true);
    setStatusMessage('');
    setErrors([]);
    if (!itemId) setFormData({});
  };

  return (
    <div>
      {statusMessage && (
        <p className={statusMessage.includes('successfully') ? 'success' : 'error'}>
          {statusMessage}
        </p>
      )}

      {!showForm && !itemId && (
        <button onClick={handleResetForm}>+ Add another {collectionName}</button>
      )}

      {loading && itemId ? (
        <div>Loading...</div>
      ) : (
        showForm && (
          <Form
            schema={dynamicSchema}
            uiSchema={uiSchema}
            formData={formData}
            onSubmit={handleSubmit}
            onError={handleError}
            validator={validator}
            liveValidate={false}
            noHtml5Validate={true}
          >
            {errors.length > 0 && (
              <div className="form-errors" style={{ color: 'red', marginBottom: '1rem' }}>
                <ul>
                  {errors.map((err, idx) => <li key={idx}>{err}</li>)}
                </ul>
              </div>
            )}
            <div style={{ marginTop: '32px' }}>
              <button
                type="submit"
                style={{
                  backgroundColor: '#f0f0f0',
                  color: '#00D563',
                  padding: '10px 16px',
                  border: 'none',
                  borderRadius: '4px',
                  cursor: 'pointer',
                  fontSize: '14px',
                  fontWeight: 'bold',
                  marginTop: '-15px'
                }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.backgroundColor = '#00D563';
                  e.currentTarget.style.color = 'white';
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.backgroundColor = 'white';
                  e.currentTarget.style.color = '#00D563';
                }}
              >
                {itemId ? 'Update' : 'Submit'}
              </button>
            </div>
          </Form>
        )
      )}
    </div>
  );
};

export default GenericForm;