import React, { useState, useEffect } from 'react';
import FormStep from './FormStep';

const FormBuilder = ({ config, onFinalSubmit, onCancel , onFieldChange }) => {
    const [formData, setFormData] = useState(config.steps[0].defaultValues || {});
    const [currentStep, setCurrentStep] = useState(0);
    const [formKey, setFormKey] = useState(Date.now());  // Create a new state for the key

    // Update formData when defaultValues change
    useEffect(() => {
        setFormData(config.steps[currentStep].defaultValues || {});
    }, [config.steps, currentStep]);

    const onInputChange = (name, value) => {
        setFormData(prevData => ({ ...prevData, [name]: value }));
        if (onFieldChange) {
            onFieldChange(name, value);
        }

    };

    const onNextStep = () => {
        setCurrentStep(prevStep => prevStep + 1);
    };

    const onPrevStep = () => {
        setCurrentStep(prevStep => prevStep - 1);
    };

    const onSubmit = (e) => {
        e.preventDefault();

        if (currentStep === config.steps.length - 1) {
            onFinalSubmit(formData);
            return;
        }

        onNextStep();
    };

    const resetForm = () => {
        setFormData(config.steps[0].defaultValues || {});
        setCurrentStep(0);
        setFormKey(Date.now());  // Reset the form by setting a new key
        if (onCancel) {
            onCancel();
        }
    };

    return (
        <form onSubmit={onSubmit} key={formKey}>
            {config.steps.map((step, index) => (
                <FormStep 
                    key={index}
                    fields={step.fields}
                    formType={config.formType}
                    formData={formData}
                    onInputChange={onInputChange}
                    visible={index === currentStep}
                />
            ))}
            <div className="form-actions">
                {currentStep > 0 && (
                    <button type="button" onClick={onPrevStep}>Previous</button>
                )}
                <button type="submit">{currentStep === config.steps.length - 1 ? 'Submit' : 'Next'}</button>
            </div>
        </form>
    );
};

export default FormBuilder;
