// src/components/common/Builder/AutoCompleteWidget.js
import React, { useState, useEffect, useRef } from 'react';
import axios from 'axios';

/**
 * Generic async autocomplete.
 * 
 * options.api         – URL to GET for both search & label lookup.
 * options.listKey    – JSON key for array (default: 'items').
 * options.searchField– query‑param name for search (default: 'q').
 * options.labelField – object key to display (default: 'name').
 * options.valueField – object key as value (default: 'id').
 * options.debounce   – ms debounce between keystrokes.
 * options.placeholder– input placeholder.
 */
const AutoCompleteWidget = ({ id, schema, value, onChange, options }) => {
  const {
    api,
    listKey = 'items',
    searchField = 'q',
    labelField = 'name',
    valueField = 'id',
    debounce = 300,
    placeholder = schema.title
  } = options;

  const [inputValue, setInputValue] = useState('');
  const [choices, setChoices]       = useState([]);
  const [loading, setLoading]       = useState(false);
  const [showList, setShowList]     = useState(false);
  const debounceRef                 = useRef();

  // Load current label if value exists
  useEffect(() => {
    if (value && api) {
      setLoading(true);
      axios.get(api, { params: { [valueField]: value } })
        .then(res => {
          const arr = res.data[listKey] || res.data;
          const hit = arr.find(i => i[valueField] === value);
          if (hit) setInputValue(hit[labelField]);
        })
        .catch(() => {})
        .finally(() => setLoading(false));
    }
  }, [value, api, listKey, valueField, labelField]);

  // Handle user typing
  const handleInput = (e) => {
    const q = e.target.value;
    setInputValue(q);
    setShowList(true);

    clearTimeout(debounceRef.current);
    debounceRef.current = setTimeout(() => {
      if (!api) return;
      setLoading(true);
      axios.get(api, { params: { [searchField]: q } })
        .then(res => {
          const arr = res.data[listKey] || res.data;
          setChoices(arr);
        })
        .catch(() => setChoices([]))
        .finally(() => setLoading(false));
    }, debounce);
  };

  // When user picks an item
  const handleSelect = (item) => {
    onChange(item[valueField]);
    setInputValue(item[labelField]);
    setShowList(false);
  };

  return (
    <div style={{ position: 'relative', width: '100%' }}>
      <input
        id={id}
        type="text"
        value={inputValue}
        placeholder={placeholder}
        onChange={handleInput}
        onFocus={() => setShowList(true)}
        onBlur={() => setTimeout(() => setShowList(false), 200)}
        style={{
          width: '100%',
          padding: '8px 12px',
          border: '1px solid #ccc',
          borderRadius: 4,
          boxSizing: 'border-box'
        }}
      />
      {showList && (
        <ul style={{
          position: 'absolute',
          top: '100%', left: 0, right: 0,
          maxHeight: 200, overflowY: 'auto',
          margin: 0, padding: 0, listStyle: 'none',
          border: '1px solid #ccc', background: '#fff', zIndex: 1000
        }}>
          {loading
            ? <li style={{ padding: 8 }}>Loading…</li>
            : (choices.length
              ? choices.map(item => (
                  <li
                    key={item[valueField]}
                    onMouseDown={() => handleSelect(item)}
                    style={{
                      padding: 8,
                      cursor: 'pointer',
                      borderBottom: '1px solid #eee'
                    }}
                  >
                    {item[labelField]}
                  </li>
                ))
              : <li style={{ padding: 8 }}>No results</li>
            )
          }
        </ul>
      )}
    </div>
  );
};

export default AutoCompleteWidget;
