import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import UserView from './UserView';
import UserUpdate from './UserUpdate';
import UserDB from './UserDB';

const UsersList = ({ accountId = '', initialSearchQuery = '', rowsPerPage = 10 }) => {
  const fetchDataUrl = accountId ? `/users/list/` : '/users/list/all/';
  const [selectedUser, setSelectedUser] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  // Define search fields including account_id filter
  const searchFields = [
    { 
      name: 'created_date_from', 
      type: 'date', 
      placeholder: 'From Date' 
    },
    { 
      name: 'created_date_to', 
      type: 'date', 
      placeholder: 'To Date' 
    },
    { 
      name: 'q', 
      type: 'text', 
      placeholder: 'Search users...', 
      defaultValue: initialSearchQuery 
    },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ]
    },
    { 
      name: 'role',
      type: 'dropdown',
      placeholder: 'Role',
      options: [
        { label: 'All', value: '' },
        { label: 'Admin', value: 'admin' },
        { label: 'Manager', value: 'manager' },
        { label: 'User', value: 'user' }
      ]
    },
    // Always show account_id filter
    {
      name: 'account_id',
      type: 'text',
      placeholder: 'Account ID',
    }
  ];

  // Transform search parameters before sending to API
  const transformSearchParams = (params) => {
    const transformed = { ...params };
    
    // If accountId is provided as prop, always include it in the search params
    // This will override any manually entered account_id in the filter
    if (accountId) {
      transformed.account_id = accountId;
    }
    
    return transformed;
  };

  const USER_HEADER_TO_KEY_MAP = {
    "User Name": { 
      key: "name", 
      label: "User Name", 
      sortable: true,
      clickable: true,
      formatter: (value) => value || '-'
    },
    "Email": { 
      key: "email", 
      label: "Email", 
      sortable: true,
      formatter: (value) => value || '-'
    },
    "Mobile": { 
      key: "mobile", 
      label: "Mobile", 
      sortable: false,
      formatter: (value) => value || '-'
    },
    "Roles": { 
      key: "roles", 
      label: "Roles", 
      sortable: false,
      formatter: (roles) => roles?.join(', ') || '-'
    },
    "Created Date": { 
      key: "created_date", 
      label: "Created Date", 
      sortable: true,
      formatter: (date) => date ? new Date(date).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
      }) : '-'
    },
    "Status": { 
      key: "is_active", 
      label: 'Status', 
      sortable: false,
      formatter: (active) => (
        <span style={{ color: active ? 'green' : 'red' }}>
          {active ? 'Active' : 'Inactive'}
        </span>
      )
    },
  };

  const transformUserData = (apiData) => {
    if (!apiData || !apiData.users) {
      console.error('Invalid API data structure', apiData);
      return { total_count: 0, users: [] };
    }
    
    return apiData;
  };

  const handleActionClick = (userId, action) => {
    console.log('Action Clicked:', userId, action);
    const actionMap = {
      view: 'view',
      edit: 'update',
      delete: 'delete',
      block: 'block'
    };

    setSelectedUser({ 
      id: userId, 
      modalType: actionMap[action.label],
      action: action.label
    });
  };

  const handleCellClick = (header, item) => {
    const mappingValue = USER_HEADER_TO_KEY_MAP[header];
    console.log('Cell Clicked:', header, item);

    if (!mappingValue) {
      console.warn('No mapping found for header:', header);
      return;
    }

    if (mappingValue.clickable) {
      console.log('Opening UserView for:', item.user_id);
      setSelectedUser({
        id: item.user_id,
        modalType: 'view',
        data: item
      });
    }
  };

  const closeModal = () => {
    console.log('Closing modal');
    setSelectedUser(null);
  };

  const refreshList = () => {
    console.log('Refreshing user list');
    setRefreshTrigger(prev => prev + 1);
  };

  const actionFormats = [
    { label: "view", modal: true, url: "/users/view/{id}" },
    { label: "edit", modal: true, url: "/users/update/{id}" },
    { label: "delete", modal: true, url: "/users/delete/{id}" },
    // { label: "block", modal: true, url: "/users/block/{id}" }
  ];

  return (
    <div className="user-list-container">
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={USER_HEADER_TO_KEY_MAP}
        dataKey="users"
        identifierKey="user_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        refreshTrigger={refreshTrigger}
        dataTransformer={transformUserData}
        searchFields={searchFields}
        transformSearchParams={transformSearchParams}
        renderCell={(value, column) => {
          if (value === undefined || value === null) {
            return <span style={{ color: '#999' }}>None</span>;
          }
          return String(value);
        }}
        tableProps={{
          striped: true,
          hover: true,
          responsive: true,
          style: { marginTop: '1rem' }
        }}
      />

      {selectedUser?.modalType === "view" && (
        <UserView 
          userId={selectedUser.id} 
          userData={selectedUser.data}
          onClose={closeModal} 
        />
      )}

      {selectedUser?.modalType === "update" && (
        <UserUpdate 
          userId={selectedUser.id}
          onClose={closeModal}
          onUpdateSuccess={refreshList}
        />
      )}

      {selectedUser?.modalType === "delete" && (
        <UserDB 
          userId={selectedUser.id}
          onClose={closeModal}
          onDeleteSuccess={refreshList}
        />
      )}

      {selectedUser?.modalType === "block" && (
        <UserDB 
          userId={selectedUser.id}
          action="block"
          onClose={closeModal}
          onBlockSuccess={refreshList}
        />
      )}
    </div>
  );
};

export default UsersList;