
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'dashboard', label: 'User Dashboard', icon: 'dashboard' },
  { id: 'account', label: 'Account', icon: 'account_balance' },
];

const UserView = ({ userId, onClose }) => {
  const [userData, setUserData] = useState(null);
  const [accountData, setAccountData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');

  useEffect(() => {
    const fetchUserData = async () => {
      try {
        setLoading(true);
        setError('');
        
        // Fetch user data
        const userRes = await axios.get(`/users/${userId}`);
        setUserData(userRes.data);
        
        // If user has an account ID, fetch account data
        if (userRes.data.account_id) {
          try {
            const accountRes = await axios.get(`/accounts/${userRes.data.account_id}`);
            setAccountData(accountRes.data);
          } catch (err) {
            console.error('Error fetching account data:', err);
            // Continue even if account data fails to load
          }
        }
        
        setLoading(false);
      } catch (err) {
        console.error('Error fetching user data:', err);
        setError('Error fetching user details.');
        setLoading(false);
      }
    };

    if (userId) {
      fetchUserData();
    }
  }, [userId]);

  const Icon = ({ name, color = '#000' }) => (
    <span
      className="material-symbols-outlined"
      style={{ fontSize: '20px', color, verticalAlign: 'middle', marginRight: '6px' }}
    >
      {name}
    </span>
  );

  const renderTabContent = () => {
    if (loading) {
      return (
        <div style={{ padding: '20px', textAlign: 'center' }}>
          Loading user data...
        </div>
      );
    }

    if (error) {
      return (
        <div style={{ padding: '20px', color: '#f44336' }}>
          {error}
        </div>
      );
    }

    switch (activeTab) {
      case 'dashboard':
        return (
          <div style={{ padding: '20px' }}>
            {/* Stats Cards */}
            <div style={{ 
              display: 'grid', 
              gridTemplateColumns: 'repeat(auto-fit, minmax(180px, 1fr))', 
              gap: '16px',
              marginBottom: '24px'
            }}>
              <div style={{ 
                textAlign: 'center', 
                padding: '16px',
                backgroundColor: '#f8f9fa',
                borderRadius: '12px',
                boxShadow: '0 2px 4px rgba(0,0,0,0.05)'
              }}>
                <div style={{ fontSize: '28px', fontWeight: 'bold', color: '#34c759', marginBottom: '8px' }}>
                  {userData?.login_count || 0}
                </div>
                <div style={{ fontSize: '14px', color: '#666' }}>Total Logins</div>
              </div>
              
              <div style={{ 
                textAlign: 'center', 
                padding: '16px',
                backgroundColor: '#f8f9fa',
                borderRadius: '12px',
                boxShadow: '0 2px 4px rgba(0,0,0,0.05)'
              }}>
                <div style={{ 
                  fontSize: '28px', 
                  fontWeight: 'bold', 
                  color: userData?.last_login ? '#34c759' : '#ff9500',
                  marginBottom: '8px' 
                }}>
                  {userData?.last_login ? 'Yes' : 'No'}
                </div>
                <div style={{ fontSize: '14px', color: '#666' }}>Has Logged In</div>
              </div>
              
              <div style={{ 
                textAlign: 'center', 
                padding: '16px',
                backgroundColor: '#f8f9fa',
                borderRadius: '12px',
                boxShadow: '0 2px 4px rgba(0,0,0,0.05)'
              }}>
                <div style={{ 
                  fontSize: '28px', 
                  fontWeight: 'bold', 
                  color: userData?.is_active ? '#34c759' : '#ff3b30',
                  marginBottom: '8px' 
                }}>
                  {userData?.is_active ? 'Active' : 'Inactive'}
                </div>
                <div style={{ fontSize: '14px', color: '#666' }}>Status</div>
              </div>
            </div>
            
            {/* User Information Card */}
            <div style={{ 
              marginBottom: '24px', 
              padding: '20px', 
              border: '1px solid #e0e0e0', 
              borderRadius: '12px',
              backgroundColor: 'white'
            }}>
              <h3 style={{ 
                margin: '0 0 16px 0', 
                fontSize: '18px', 
                fontWeight: '600',
                color: '#333',
                paddingBottom: '12px',
                borderBottom: '1px solid #f0f0f0'
              }}>
                User Information
              </h3>
              <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '16px' }}>
                <div>
                  <div style={{ marginBottom: '12px' }}>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>User ID</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>{userData?.user_id || userId}</div>
                  </div>
                  <div style={{ marginBottom: '12px' }}>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Name</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>{userData?.name || 'N/A'}</div>
                  </div>
                  <div style={{ marginBottom: '12px' }}>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Email</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>{userData?.email || 'N/A'}</div>
                  </div>
                  <div>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Mobile</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>{userData?.mobile || 'N/A'}</div>
                  </div>
                </div>
                <div>
                  <div style={{ marginBottom: '12px' }}>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Roles</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>
                      {Array.isArray(userData?.roles) ? userData.roles.join(', ') : userData?.roles || 'N/A'}
                    </div>
                  </div>
                  <div style={{ marginBottom: '12px' }}>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Created Date</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>
                      {userData?.created_date ? new Date(userData.created_date).toLocaleDateString() : 'N/A'}
                    </div>
                  </div>
                  <div>
                    <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Last Login</div>
                    <div style={{ fontSize: '15px', fontWeight: '500' }}>
                      {userData?.last_login ? new Date(userData.last_login).toLocaleString() : 'Never'}
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Associated Account Card */}
            {accountData && (
              <div style={{ 
                padding: '20px', 
                border: '1px solid #e0e0e0', 
                borderRadius: '12px',
                backgroundColor: 'white'
              }}>
                <h3 style={{ 
                  margin: '0 0 16px 0', 
                  fontSize: '18px', 
                  fontWeight: '600',
                  color: '#333',
                  paddingBottom: '12px',
                  borderBottom: '1px solid #f0f0f0'
                }}>
                  Associated Account
                </h3>
                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '16px' }}>
                  <div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Account Name</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.account_name}</div>
                    </div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Account Type</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.account_type}</div>
                    </div>
                  </div>
                  <div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Industry</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.industry}</div>
                    </div>
                    <div>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Status</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>
                        <span style={{ 
                          backgroundColor: accountData.is_active ? '#34c759' : '#ff3b30',
                          color: '#fff',
                          padding: '4px 10px',
                          borderRadius: '12px',
                          fontSize: '0.8em',
                          display: 'inline-block'
                        }}>
                          {accountData.is_active ? "Active" : "Inactive"}
                        </span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        );

      case 'account':
        return (
          <div style={{ padding: '20px' }}>
            {accountData ? (
              <>
                {/* Account Header */}
                <div style={{ 
                  display: 'flex', 
                  alignItems: 'center', 
                  marginBottom: '24px',
                  padding: '20px',
                  backgroundColor: 'white',
                  borderRadius: '12px',
                  border: '1px solid #e0e0e0'
                }}>
                  <div style={{
                    width: '60px',
                    height: '60px',
                    borderRadius: '50%',
                    backgroundColor: '#e0e0e0',
                    color: '#000',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontSize: '24px',
                    fontWeight: 'bold',
                    marginRight: '16px'
                  }}>
                    {accountData.account_name?.charAt(0)?.toUpperCase() || 'A'}
                  </div>
                  <div>
                    <h2 style={{ margin: '0 0 4px 0', fontSize: '20px', fontWeight: '600' }}>
                      {accountData.account_name}
                    </h2>
                    <p style={{ margin: 0, color: '#666', fontSize: '14px' }}>
                      {accountData.industry} • {accountData.account_type}
                    </p>
                  </div>
                </div>
                
                {/* Account Details */}
                <div style={{ 
                  display: 'grid', 
                  gridTemplateColumns: '1fr 1fr', 
                  gap: '20px', 
                  marginBottom: '24px' 
                }}>
                  <div style={{ 
                    padding: '20px', 
                    border: '1px solid #e0e0e0', 
                    borderRadius: '12px',
                    backgroundColor: 'white'
                  }}>
                    <h3 style={{ 
                      margin: '0 0 16px 0', 
                      fontSize: '16px', 
                      fontWeight: '600',
                      color: '#333',
                      paddingBottom: '12px',
                      borderBottom: '1px solid #f0f0f0'
                    }}>
                      Account Details
                    </h3>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Account Name</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.account_name}</div>
                    </div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Account Type</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.account_type}</div>
                    </div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Industry</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.industry}</div>
                    </div>
                    <div>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Status</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>
                        <span style={{ 
                          backgroundColor: accountData.is_active ? '#34c759' : '#ff3b30',
                          color: '#fff',
                          padding: '4px 10px',
                          borderRadius: '12px',
                          fontSize: '0.8em',
                          display: 'inline-block'
                        }}>
                          {accountData.is_active ? "Active" : "Inactive"}
                        </span>
                      </div>
                    </div>
                  </div>
                  
                  <div style={{ 
                    padding: '20px', 
                    border: '1px solid #e0e0e0', 
                    borderRadius: '12px',
                    backgroundColor: 'white'
                  }}>
                    <h3 style={{ 
                      margin: '0 0 16px 0', 
                      fontSize: '16px', 
                      fontWeight: '600',
                      color: '#333',
                      paddingBottom: '12px',
                      borderBottom: '1px solid #f0f0f0'
                    }}>
                      Subscription
                    </h3>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Subscription Status</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.subscription_status || 'N/A'}</div>
                    </div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Subscription Type</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.subscription_type || 'N/A'}</div>
                    </div>
                    <div style={{ marginBottom: '12px' }}>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Created Date</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>
                        {accountData.created_date ? new Date(accountData.created_date).toLocaleDateString() : 'N/A'}
                      </div>
                    </div>
                    <div>
                      <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Last Updated</div>
                      <div style={{ fontSize: '15px', fontWeight: '500' }}>
                        {accountData.updated_date ? new Date(accountData.updated_date).toLocaleDateString() : 'N/A'}
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* Contact Information */}
                <div style={{ 
                  padding: '20px', 
                  border: '1px solid #e0e0e0', 
                  borderRadius: '12px',
                  backgroundColor: 'white'
                }}>
                  <h3 style={{ 
                    margin: '0 0 16px 0', 
                    fontSize: '16px', 
                    fontWeight: '600',
                    color: '#333',
                    paddingBottom: '12px',
                    borderBottom: '1px solid #f0f0f0'
                  }}>
                    Contact Information
                  </h3>
                  <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '16px' }}>
                    <div>
                      <div style={{ marginBottom: '12px' }}>
                        <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Contact Person</div>
                        <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.name || 'N/A'}</div>
                      </div>
                      <div>
                        <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Email</div>
                        <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.email || 'N/A'}</div>
                      </div>
                    </div>
                    <div>
                      <div style={{ marginBottom: '12px' }}>
                        <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Mobile</div>
                        <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.mobile || 'N/A'}</div>
                      </div>
                      <div>
                        <div style={{ fontSize: '13px', color: '#666', marginBottom: '4px' }}>Website</div>
                        <div style={{ fontSize: '15px', fontWeight: '500' }}>{accountData.website || 'N/A'}</div>
                      </div>
                    </div>
                  </div>
                </div>
              </>
            ) : (
              <div style={{ 
                padding: '40px', 
                textAlign: 'center', 
                color: '#666',
                backgroundColor: '#f8f9fa',
                borderRadius: '12px',
                border: '1px dashed #e0e0e0'
              }}>
                <span className="material-symbols-outlined" style={{ fontSize: '48px', marginBottom: '16px', display: 'block', color: '#ccc' }}>
                  account_balance
                </span>
                <h3 style={{ margin: '0 0 8px 0', fontSize: '18px', fontWeight: '500' }}>
                  {userData?.account_id ? 
                    'Unable to load account information' : 
                    'No associated account'
                  }
                </h3>
                <p style={{ margin: 0, fontSize: '14px' }}>
                  {userData?.account_id ? 
                    'There was an error loading the account details.' : 
                    'This user is not associated with any account.'
                  }
                </p>
              </div>
            )}
          </div>
        );

      default:
        return null;
    }
  };

  return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          padding: '16px',
          justifyContent: 'space-between',
          borderBottom: '1px solid #eee',
          backgroundColor: 'white'
        }}>
          <div style={{ display: 'flex', alignItems: 'center' }}>
            <div style={{
              width: '60px',
              height: '60px',
              borderRadius: '50%',
              backgroundColor: '#e0e0e0',
              color: '#000',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              fontSize: '20px',
              fontWeight: 'bold',
              marginRight: '16px'
            }}>
              {userData?.name?.charAt(0)?.toUpperCase() || 'U'}
            </div>
            <div>
              <div style={{ fontWeight: '600', fontSize: '20px', color: 'black' }}>
                {userData?.name || 'User'}
              </div>
              <div style={{ fontSize: '14px', color: '#666', marginTop: '4px' }}>
                ID: #{userData?.user_id || userId}
              </div>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '0',
          padding: '0 16px',
          overflowX: 'auto',
          scrollbarWidth: 'none',
          msOverflowStyle: 'none',
          backgroundColor: 'white'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '16px 20px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '600' : '400',
                color: activeTab === tab.id ? '#34c759' : '#666',
                position: 'relative',
                display: 'flex',
                alignItems: 'center',
                gap: '8px',
                whiteSpace: 'nowrap',
                flexShrink: 0
              }}
              onClick={() => setActiveTab(tab.id)}
            >
              <Icon name={tab.icon} color={activeTab === tab.id ? '#34c759' : '#666'} />
              {tab.label}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-1px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Content */}
        <div style={{ 
          maxHeight: 'calc(100vh - 200px)', 
          overflowY: 'auto',
          padding: '0',
          backgroundColor: '#f8f9fa'
        }}>
          {renderTabContent()}
        </div>
      </div>
    </Modal>
  );
};

export default UserView;