// TicketView.js
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TicketView = ({ ticketId, onClose }) => {
    const [replies, setReplies] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');

    useEffect(() => {
        const fetchData = async () => {
            try {
                setLoading(true);
                const repliesResponse = await axios.get(`/ticket/reply/list/${ticketId}`);
                setReplies(repliesResponse.data.replies || []);
                setLoading(false);
            } catch (err) {
                console.error("Error fetching ticket replies:", err);
                setError('Error fetching ticket replies.');
                setLoading(false);
            }
        };

        fetchData();
    }, [ticketId]);

    if (loading) {
        return (
            <Modal onClose={onClose}>
                <div style={{ padding: '20px', textAlign: 'center' }}>Loading ticket replies...</div>
            </Modal>
        );
    }

    if (error) {
        return (
            <Modal onClose={onClose}>
                <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
            </Modal>
        );
    }

    return (
        <Modal onClose={onClose}>
            <div style={{ maxWidth: '800px', padding: '20px' }}>
                <h3 style={{ marginBottom: '20px', color: '#333' }}>
                    Replies for Ticket #{ticketId}
                </h3>

                <div style={{ marginTop: '30px' }}>
                    <h4>Replies</h4>
                    {replies.length > 0 ? (
                        replies.map((reply, index) => (
                            <div key={index} style={{
                                border: '1px solid #eee',
                                borderRadius: '8px',
                                padding: '15px',
                                marginBottom: '15px'
                            }}>
                                <div style={{ marginBottom: '10px' }}>
                                    <strong>Reply #{index + 1}</strong>
                                    {reply.status && (
                                        <span style={{ 
                                            marginLeft: '10px',
                                            padding: '3px 8px',
                                            borderRadius: '4px',
                                            backgroundColor: reply.status === 'resolved' ? '#4caf50' : '#ff9800',
                                            color: 'white',
                                            fontSize: '0.8em'
                                        }}>
                                            {reply.status}
                                        </span>
                                    )}
                                </div>
                                <div style={{ marginBottom: '10px' }}>
                                    <p>{reply.description}</p>
                                </div>
                                <div style={{ fontSize: '0.8em', color: '#666' }}>
                                    {new Date(reply.created_date).toLocaleString()}
                                </div>
                            </div>
                        ))
                    ) : (
                        <div style={{ 
                            border: '1px solid #eee',
                            borderRadius: '8px',
                            padding: '20px',
                            textAlign: 'center',
                            color: '#666'
                        }}>
                            No replies found for this ticket
                        </div>
                    )}
                </div>
            </div>
        </Modal>
    );
};

export default TicketView;