// TicketList.js
import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import TicketView from './TicketView';
import TicketUpdate from './TicketUpdate';
import TicketDB from './TicketDB';

const TICKETS_HEADER_TO_KEY_MAP = {
    "Subject": { 
        key: "subject", 
        label: "Subject", 
        sortable: true, 
        clickable: true 
    },
    "Description": { key: "description", label: "Description", sortable: false },
    "Status": { key: "status", label: "Status", sortable: false },
    "Created Date": { key: "created_date", label: "Created Date", sortable: true },
};

const TicketList = ({ initialSearchQuery = '', rowsPerPage = 10, accountId = null }) => {
    const [searchParams, setSearchParams] = useState({
        skip: 0,
        limit: rowsPerPage,
        q: initialSearchQuery
    });

    const [selectedTicket, setSelectedTicket] = useState(null);

    const searchFields = [
        { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
        {
            name: 'status',
            type: 'dropdown',
            placeholder: 'Status',
            options: [
                { label: 'All', value: '' },
                { label: 'Open', value: 'open' },
                { label: 'In Progress', value: 'in_progress' },
                { label: 'Resolved', value: 'resolved' },
                { label: 'Closed', value: 'closed' }
            ]
        },
        { name: 'created_date_from', type: 'date', placeholder: 'From Date' },
        { name: 'created_date_to', type: 'date', placeholder: 'To Date' },
        // Add account_id filter
        {
            name: 'account_id',
            type: 'text',
            placeholder: 'Account ID',
        }
    ];

    // Transform search parameters before sending to API
    const transformSearchParams = (params) => {
        const transformed = { ...params };
        
        // If accountId is provided as prop, always include it in the search params
        // This will override any manually entered account_id in the filter
        if (accountId) {
            transformed.account_id = accountId;
        }
        
        return transformed;
    };

    const actionFormats = [
        { label: "view", modal: true, url: "/tickets/view/{id}" },
        { label: "edit", modal: true, url: "/tickets/update/{id}" },
        { label: "delete", modal: true, url: "/tickets/delete/{id}" },
    ];

    const handleActionClick = (ticketId, action) => {
        if (["view", "edit", "delete"].includes(action.label)) {
            setSelectedTicket({ ticketId, modalType: action.label });
        }
    };

    const handleCellClick = (header, item) => {
        if (header === "Subject") {
            const ticketId = item.ticket_id || item.id;
            if (ticketId) {
                setSelectedTicket({
                    ticketId: ticketId,
                    modalType: 'view'
                });
            }
        }
    };

    const handleSearch = (params) => {
        setSearchParams(prev => ({
            ...prev,
            q: params.q || '',
            created_date_from: params.created_date_from || '',
            created_date_to: params.created_date_to || '',
            status: params.status || '',
            account_id: params.account_id || '',
            skip: 0
        }));
    };

    const handlePageChange = (page) => {
        setSearchParams(prev => ({
            ...prev,
            skip: (page - 1) * rowsPerPage
        }));
    };

    const closeModal = () => setSelectedTicket(null);

    const buildFetchUrl = () => {
        let url = `/ticket/list/?skip=${searchParams.skip}&limit=${searchParams.limit}`;
        if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
        if (searchParams.created_date_from) url += `&from=${searchParams.created_date_from}`;
        if (searchParams.created_date_to) url += `&to=${searchParams.created_date_to}`;
        if (searchParams.status) url += `&status=${searchParams.status}`;
        if (searchParams.account_id) url += `&account_id=${searchParams.account_id}`;
        return url;
    };

    return (
        <div>
            <SearchPage
                fetchDataUrl={buildFetchUrl()}
                ListComponent={List}
                itemKeyMapping={TICKETS_HEADER_TO_KEY_MAP}
                dataKey="tickets"
                identifierKey="ticket_id"
                initialSearchQuery={initialSearchQuery}
                rowsPerPage={rowsPerPage}
                actionFormats={actionFormats}
                onActionClick={handleActionClick}
                onCellClick={handleCellClick}
                searchFields={searchFields}
                transformSearchParams={transformSearchParams}
                onSearch={handleSearch}
                onPageChange={handlePageChange}
            />

            {selectedTicket?.modalType === "view" && (
                <TicketView
                    ticketId={selectedTicket.ticketId}
                    onClose={closeModal}
                />
            )}

            {selectedTicket?.modalType === "edit" && (
                <TicketUpdate
                    ticketId={selectedTicket.ticketId}
                    onClose={closeModal}
                    onUpdateSuccess={() => console.log("Ticket updated")}
                />
            )}

            {selectedTicket?.modalType === "delete" && (
                <TicketDB
                    ticketId={selectedTicket.ticketId}
                    onClose={closeModal}
                    onDeleteSuccess={() => console.log("Ticket deleted")}
                />
            )}
        </div>
    );
};

export default TicketList;