
import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import SubscriptionView from './SubscriptionView';
import SubscriptionUpdate from './SubscriptionUpdate';
import SubscriptionDB from './SubscriptionDB';

const STATUS_COLORS = {
  new: '#ff9800',
  active: '#4caf50',
  cancelled: '#f44336',
  expired: '#9e9e9e',
  paused: '#673ab7',
  unpaid: '#ff5722',
  suspended: '#607d8b',
};

const SUBS_HEADER_TO_KEY_MAP = {
  'Plan Name': { key: 'plan_name', label: 'Plan Name', sortable: true },
  'Per Agent': { key: 'peragent_cost', label: 'Per Agent', sortable: true, formatter: (v) => (v != null ? `$${Number(v).toFixed(2)}` : '-') },
  'Total': { key: 'total_cost', label: 'Total', sortable: true, formatter: (v) => (v != null ? `$${Number(v).toFixed(2)}` : '-') },
  'Start': { key: 'start_date', label: 'Start', sortable: true, formatter: (d) => d ? new Date(d).toLocaleString() : '-' },
  'Renew From': { key: 'renew_start_date', label: 'Renew From', sortable: true, formatter: (d) => d ? new Date(d).toLocaleString() : '-' },
  'Renew To': { key: 'renew_end_date', label: 'Renew To', sortable: true, formatter: (d) => d ? new Date(d).toLocaleString() : '-' },
  'Status': {
    key: 'status', label: 'Status', sortable: true,
    formatter: (v) => {
      const bg = STATUS_COLORS[(v || '').toLowerCase()] || '#9e9e9e';
      return (
        <span style={{ backgroundColor: bg, color: '#fff', padding: '4px 10px', borderRadius: 12, fontSize: '0.8em', display: 'inline-block' }}>
          {v || '-'}
        </span>
      );
    }
  },
};

const SubscriptionsList = ({ initialSearchQuery = '', rowsPerPage = 10, accountId, isEmbedded = false }) => {
  const [modal, setModal] = useState(null);
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery,
    status: '',
  });

  const actionFormats = [
    { label: 'view', modal: true, url: '/subscriptions/view/{id}' },
    { label: 'edit', modal: true, url: '/subscriptions/update/{id}' },
    { label: 'delete', modal: true, url: '/subscriptions/delete/{id}' },
  ];

  const onActionClick = (id, action) => {
    if (action.label === 'edit') setModal({ type: 'update', id });
    else if (action.label === 'delete') setModal({ type: 'delete', id });
    else setModal({ type: 'view', id });
  };

  const onCellClick = (_header, item) => setModal({ type: 'view', id: item.id, data: item });
  const close = () => setModal(null);

  const onSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      status: params.status || '',
      skip: 0,
    }));
  };

  const onPageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage,
    }));
  };

  const buildUrl = () => {
    let url = `crud/subscriptions?skip=${searchParams.skip}&limit=${searchParams.limit}`;
    
    // Add account filter if accountId is provided (for embedded view)
    if (accountId) {
      url += `&account_id=${accountId}`;
    }
    
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.status) url += `&status=${encodeURIComponent(searchParams.status)}`;
    return url;
  };

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'new', value: 'new' },
        { label: 'active', value: 'active' },
        { label: 'cancelled', value: 'cancelled' },
        { label: 'expired', value: 'expired' },
        { label: 'paused', value: 'paused' },
        { label: 'unpaid', value: 'unpaid' },
        { label: 'suspended', value: 'suspended' },
      ],
    },
  ];

  return (
    <div>
      <SearchPage
        fetchDataUrl={buildUrl()}
        ListComponent={List}
        itemKeyMapping={SUBS_HEADER_TO_KEY_MAP}
        dataKey="items"
        identifierKey="id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={onActionClick}
        onCellClick={onCellClick}
        searchFields={searchFields}
        onSearch={onSearch}
        onPageChange={onPageChange}
      />

      {modal?.type === 'view' && (
        <SubscriptionView
          subscriptionId={modal.id}
          subscriptionData={modal.data}
          onClose={close}
          onEdit={() => setModal({ type: 'update', id: modal.id })}
          onDelete={() => setModal({ type: 'delete', id: modal.id })}
        />
      )}

      {modal?.type === 'update' && (
        <SubscriptionUpdate
          subscriptionId={modal.id}
          onClose={close}
          onUpdateSuccess={() => window.location.reload()}
        />
      )}

      {modal?.type === 'delete' && (
        <SubscriptionDB
          subscriptionId={modal.id}
          onClose={close}
          onDeleteSuccess={() => window.location.reload()}
        />
      )}
    </div>
  );
};

export default SubscriptionsList;
