// SubscriptionView.js
import React, { useEffect, useState } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'overview', label: 'Dashboard', icon: 'dashboard' },
  { id: 'account', label: 'Account Info', icon: 'info' },
  { id: 'payment', label: 'Payments', icon: 'payments' },
];

const Icon = ({ name, color = '#000' }) => (
  <span
    className="material-symbols-outlined"
    style={{ fontSize: '24px', color, verticalAlign: 'middle', marginRight: '6px' }}
  >
    {name}
  </span>
);

const Header = ({ title, subtitle }) => (
  <div style={{ display: 'flex', alignItems: 'center', padding: '8px 12px', gap: 12 }}>
    <div
      style={{
        width: 40,
        height: 40,
        borderRadius: 999,
        background: '#eee',
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        fontWeight: 700,
        color: '#333',
        marginRight: 16,
      }}
    >
      {title?.charAt(0)?.toUpperCase() || 'S'}
    </div>
    <div>
      <div style={{ fontWeight: 600, fontSize: 20, color: 'black' }}>
        {title || 'Plan'}
      </div>
      <div style={{ fontSize: 14, color: '#666', marginTop: 4 }}>{subtitle || ''}</div>
    </div>
  </div>
);

const TabBar = ({ tabs, active, onChange }) => (
  <div
    style={{
      display: 'flex',
      borderBottom: '1px solid #e0e0e0',
      borderTop: '1px solid #e0e0e0', 
      marginBottom: 0,
      padding: '0 10px',
      overflowX: 'auto',
      scrollbarWidth: 'none',
      msOverflowStyle: 'none',
    }}
  >
    {tabs.map((tab) => (
      <button
        key={tab.id}
        style={{
          padding: '12px 16px',
          border: 'none',
          backgroundColor: 'transparent',
          cursor: 'pointer',
          fontSize: '14px',
          fontWeight: active === tab.id ? 600 : 400,
          color: active === tab.id ? '#34c759' : '#666',
          position: 'relative',
          display: 'flex',
          alignItems: 'center',
          gap: 8,
          whiteSpace: 'nowrap',
          flexShrink: 0,
        }}
        onClick={() => onChange(tab.id)}
      >
        <Icon name={tab.icon} color={active === tab.id ? '#34c759' : '#666'} />
        {tab.label}
        {active === tab.id && (
          <div
            style={{
              position: 'absolute',
              bottom: -1,
              left: 0,
              right: 0,
              height: 2,
              backgroundColor: '#34c759',
            }}
          />
        )}
      </button>
    ))}
  </div>
);

const Row = ({ label, value }) => (
  <div style={{ marginBottom: 8 }}>
    <strong>{label}:</strong> {value ?? '-'}
  </div>
);

const SubscriptionView = ({ subscriptionId, subscriptionData: initial, onClose, onEdit, onDelete }) => {
  const [data, setData] = useState(initial || null);
  const [loading, setLoading] = useState(!initial);
  const [error, setError] = useState('');
  const [active, setActive] = useState('overview');
  const [accountIndustry, setAccountIndustry] = useState('');

  useEffect(() => {
    const load = async () => {
      try {
        setLoading(true);
        setError('');
        if (!initial) {
          const res = await axios.get('crud/subscriptions', {
            params: { skip: 0, limit: 50, q: subscriptionId },
          });
          const found = (res.data?.items || []).find((x) => x.id === subscriptionId) || null;
          setData(found);
        }
      } catch (e) {
        setError('Failed to load subscription.');
      } finally {
        setLoading(false);
      }
    };
    if (subscriptionId) load();
  }, [subscriptionId, initial]);

  const actions = (
    <div style={{ display: 'flex', gap: 8 }}>
      <button className="btn btn-secondary" onClick={onEdit}>
        Edit
      </button>
      <button className="btn btn-danger" onClick={onDelete}>
        Delete
      </button>
    </div>
  );

  const Overview = () => (
    <div style={{ padding: 16 }}>
      <Row label="Status" value={data.status} />
      <Row label="Agents (Plan)" value={data.subscription_agents_count} />
      <Row label="Active Agents" value={data.active_agents_count} />
      <Row
        label="Per-Agent Cost"
        value={data.peragent_cost != null ? `$${Number(data.peragent_cost).toFixed(2)}` : '-'}
      />
      <Row
        label="Total Cost"
        value={data.total_cost != null ? `$${Number(data.total_cost).toFixed(2)}` : '-'}
      />
      <Row label="Start" value={data.start_date ? new Date(data.start_date).toLocaleString() : '-'} />
      <Row
        label="Renew Start"
        value={data.renew_start_date ? new Date(data.renew_start_date).toLocaleString() : '-'}
      />
      <Row
        label="Renew End"
        value={data.renew_end_date ? new Date(data.renew_end_date).toLocaleString() : '-'}
      />
      <Row label="Discount" value={data.discount ?? 0} />
      <Row label="Discount Reason" value={data.discount_reason || '-'} />
      <Row label="ID" value={data.id} />
      <Row label="Subscription Type ID" value={data.subscription_type_id} />
    </div>
  );

  const AccountInfo = () => (
    <div style={{ padding: 16 }}>
      <Row label="Account ID" value={data.account_id} />
      <Row label="User ID" value={data.user_id} />
    </div>
  );

  const Payment = () => (
    <div style={{ padding: 16 }}>
      <Row label="Stripe Subscription ID" value={data.stripe_subscription_id || '-'} />
      <Row
        label="Per-Agent Cost"
        value={data.peragent_cost != null ? `$${Number(data.peragent_cost).toFixed(2)}` : '-'}
      />
      <Row
        label="Total Cost"
        value={data.total_cost != null ? `$${Number(data.total_cost).toFixed(2)}` : '-'}
      />
      <Row label="Discount" value={data.discount ?? 0} />
      <Row label="Discount Reason" value={data.discount_reason || '-'} />
    </div>
  );

  const headerTitle = data?.plan_name || 'Plan';
  const headerSubtitle = data?.status ? `Status: ${data.status}` : '';


  return (
  <Modal onClose={onClose} actions={actions}>
    {loading ? (
      <div style={{ padding: 16 }}>Loading…</div>
    ) : error ? (
      <div style={{ padding: 16, color: 'crimson' }}>{error}</div>
    ) : !data ? (
      <div style={{ padding: 16 }}>Not found.</div>
    ) : (
      <div style={{ padding: 0 }}>
        {/* 👇 plan_name som title, status rett under */}
        <Header title={headerTitle} subtitle={headerSubtitle} />
        <TabBar tabs={TAB_CONFIG} active={active} onChange={setActive} />
        <div style={{ maxHeight: 'calc(100vh - 200px)', overflowY: 'auto', padding: 0 }}>
          {active === 'overview' && <Overview />}
          {active === 'account' && <AccountInfo />}
          {active === 'payment' && <Payment />}
        </div>
      </div>
    )}
  </Modal>
);
};

export default SubscriptionView;
