import React, { useEffect, useMemo, useRef, useState } from 'react';
import { withTheme } from '@rjsf/core';
import MaterialUITheme from '@rjsf/material-ui';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';

const Form = withTheme(MaterialUITheme);
const DEFAULT_STATUS = ['new','active','unpaid','expired','cancelled','suspended','paused'];

export default function SubscriptionForm({ initialData = {}, isEdit = false, onSubmit }) {
  // ----- Local state -----
  const [formData, setFormData] = useState({ ...initialData });
  const [errorMessage, setErrorMessage] = useState('');
  const [successMessage, setSuccessMessage] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitted, setSubmitted] = useState(false);

  // Account search state
  const [accountQuery, setAccountQuery] = useState('');
  const [accountOptions, setAccountOptions] = useState([]);
  const [isLoadingAccounts, setIsLoadingAccounts] = useState(false);
  const [selectedAccount, setSelectedAccount] = useState(
    initialData?.account_id
      ? { account_id: initialData.account_id, account_name: initialData.account_name || '' }
      : null
  );

  const debounceTimer = useRef(null);

  // ----- Schema (user_id removed; keep account_id hidden) -----
  const schema = useMemo(() => ({
    
    type: 'object',
    required: ['subscription_type_id','subscription_agents_count','peragent_cost','status'],
    properties: {
      id: { type: 'string', title: 'Subscription ID' },

      // Hidden: set by account picker
      account_id: { type: 'string', title: 'Account ID' },

      subscription_type_id: { type: 'integer', title: 'Type (ID)', default: 1 },
      subscription_agents_count: { type: 'integer', title: 'Agents Count', default: 1, minimum: 0 },
      active_agents_count: { type: 'integer', title: 'Active Agents', default: 0, minimum: 0 },
      peragent_cost: { type: 'number', title: 'Per Agent Cost', default: 60 },
      total_cost: { type: 'number', title: 'Total Cost', readOnly: true },
      discount: { type: 'number', title: 'Discount', default: 0, minimum: 0 },
      discount_reason: { type: 'string', title: 'Discount Reason' },
      stripe_subscription_id: { type: 'string', title: 'Stripe Subscription ID' },
      status: { type: 'string', title: 'Status', enum: DEFAULT_STATUS, enumNames: DEFAULT_STATUS },
      start_date: { type: 'string', title: 'Start Date', format: 'date-time' },
      renew_start_date: { type: 'string', title: 'Renew Start', format: 'date-time' },
      renew_end_date: { type: 'string', title: 'Renew End', format: 'date-time' },
    }
  }), []);

  const uiSchema = useMemo(() => ({
    ...(isEdit ? { id: { 'ui:widget': 'hidden' } } : {}),
    account_id: { 'ui:widget': 'hidden' },   // we render a custom Account Name selector
    total_cost: { 'ui:disabled': true },
  }), [isEdit]);

  // ----- Helpers -----
  const recalc = (data) => {
    const agents = Number(data.subscription_agents_count || 0);
    const per = Number(data.peragent_cost || 0);
    const disc = Number(data.discount || 0);
    const total = Math.max(0, agents * per - disc);
    return { ...data, total_cost: Number.isFinite(total) ? total : 0 };
  };

  const handleChange = ({ formData }) => setFormData(recalc(formData));

  // ----- Account search -----
  const fetchAccounts = async (q) => {
    if (!q || q.trim().length < 1) {
      setAccountOptions([]);
      return;
    }
    setIsLoadingAccounts(true);
    try {
      const params = new URLSearchParams({
        skip: '0',
        limit: '10',
        q: q.trim(),
        subscription_status: '',
        reg_date_from: '',
        reg_date_to: '',
        workforce_count_range: '',
        fleet_count_range: '',
        revenue_range: '',
      });
      const url = `accounts/?${params.toString()}`;
      const res = await axios.get(url);
      setAccountOptions(res?.data?.accounts || []);
    } catch {
      setAccountOptions([]);
    } finally {
      setIsLoadingAccounts(false);
    }
  };

  const fetchAccountById = async (id) => {
    if (!id) return null;
    try {
      const url = `accounts/?account_id=${encodeURIComponent(id)}&skip=0&limit=1&q=&subscription_status=&reg_date_from=&reg_date_to=&workforce_count_range=&fleet_count_range=&revenue_range=`;
      const res = await axios.get(url);
      const acc = (res?.data?.accounts || [])[0];
      return acc || null;
    } catch {
      return null;
    }
  };

  // If editing and we only have account_id, hydrate the account_name
  useEffect(() => {
    (async () => {
      if (selectedAccount?.account_id && !selectedAccount?.account_name) {
        const acc = await fetchAccountById(selectedAccount.account_id);
        if (acc) {
          setSelectedAccount({
            account_id: acc.account_id,
            account_name: acc.account_name || acc.title || '',
          });
        }
      }
    })();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // keep formData.account_id in sync with selected account
  useEffect(() => {
    if (selectedAccount?.account_id) {
      setFormData(prev => ({ ...prev, account_id: selectedAccount.account_id }));
    } else {
      setFormData(prev => {
        const copy = { ...prev };
        delete copy.account_id;
        return copy;
      });
    }
  }, [selectedAccount]);

  const onSelectAccount = (acc) => {
    setSelectedAccount({
      account_id: acc.account_id,
      account_name: acc.account_name || acc.title || '',
    });
    setAccountQuery(acc.account_name || acc.title || '');
    setAccountOptions([]); // collapse list after selection
  };

  const onSearchInput = (e) => {
    const q = e.target.value;
    setAccountQuery(q);
    if (debounceTimer.current) clearTimeout(debounceTimer.current);
    debounceTimer.current = setTimeout(() => fetchAccounts(q), 300);
  };

  // ----- Submit -----
  const handleSubmit = async ({ formData }) => {
    if (isSubmitting) return;
    setIsSubmitting(true);
    setErrorMessage('');
    try {
      const payload = recalc(formData);
      const url = isEdit && payload.id
        ? `crud/subscriptions/${payload.id}`
        : `crud/subscriptions`;
      const method = isEdit && payload.id ? 'put' : 'post';
      const { data, status } = await axios({ url, method, data: payload });
      if (status === 200 || status === 201) {
        setSuccessMessage(isEdit ? 'Subscription updated successfully.' : 'Subscription created successfully.');
        setSubmitted(true);
        onSubmit && onSubmit(data);
      } else {
        setErrorMessage('Unexpected response from server.');
      }
    } catch (err) {
      setErrorMessage(err?.response?.data?.message || 'Failed to submit subscription.');
    } finally {
      setIsSubmitting(false);
    }
  };

  const resetForm = () => {
    setFormData({});
    setSelectedAccount(null);
    setAccountQuery('');
    setErrorMessage('');
    setSuccessMessage('');
    setSubmitted(false);
  };

  // ----- Render -----
  return (
    <div style={{ maxWidth: '600px', padding: '20px' }}>
      {submitted ? (
        <div>
          <h2>{isEdit ? 'Subscription Updated' : 'Subscription Created'}</h2>
          {successMessage && <p style={{ color: 'green' }}>{successMessage}</p>}
          <button onClick={resetForm}>Add Another</button>
        </div>
      ) : (
        <>
          {errorMessage && <p style={{ color: 'red' }}>{errorMessage}</p>}

          {/* Account Name typeahead (shows ONLY account name) */}
          <div style={{ marginBottom: 12, position: 'relative' }}>
            <label style={{ display: 'block', fontWeight: 600, marginBottom: 6 }}>
              Account Name
            </label>
            <input
              type="text"
              placeholder="Type to search account…"
              value={accountQuery}
              onChange={onSearchInput}
              style={{
                width: '100%',
                padding: 10,
                border: '1px solid #dcdcdc',
                borderRadius: 8,
                outline: 'none'
              }}
            />

            {/* Loading note */}
            {isLoadingAccounts && (
              <div style={{ fontSize: 12, color: '#666', marginTop: 6 }}>Searching…</div>
            )}

            
           {/* Dropdown with ONLY account name */}
{!isLoadingAccounts && accountOptions.length > 0 && (
  <div
    style={{
      position: 'absolute',
      zIndex: 20,
      left: 0,
      right: 0,
      marginTop: 6,
      border: '1px solid #e6e6e6',
      borderRadius: 6,
      background: '#fff',
      maxHeight: 240,
      overflowY: 'auto',
      boxShadow: '0 8px 20px rgba(0,0,0,0.06)',
    }}
  >
    {accountOptions.map((acc) => {
      const label = acc.account_name || acc.title || acc.name || acc.account_id;
      const active = selectedAccount?.account_id === acc.account_id;
      return (
        <button
          key={acc.account_id}
          type="button"
          onClick={() => onSelectAccount(acc)}
          style={{
            width: '100%',
            textAlign: 'left',
            padding: '12px 14px',
            border: 'none',
            background: active ? '#f9f9f9' : '#fff',
            cursor: 'pointer',
            borderBottom: '1px solid #f3f3f3',
            color: '#000',              // ⬅ text always black
            fontWeight: active ? 600 : 400
          }}
          onMouseOver={e => (e.currentTarget.style.background = '#f5f5f5')}
          onMouseOut={e => (e.currentTarget.style.background = active ? '#f9f9f9' : '#fff')}
        >
          {label}
        </button>
      );
    })}
  </div>
)}

          </div>

          <Form
            schema={schema}
            uiSchema={uiSchema}
            formData={formData}
            onChange={handleChange}
            onSubmit={handleSubmit}
            validator={validator}
            liveValidate={false}
            noHtml5Validate
          >
            <div style={{ textAlign: 'center', marginTop: '10px' }}>
              <button type="submit" disabled={isSubmitting}>
                {isSubmitting ? 'Submitting...' : (isEdit ? 'Update Subscription' : 'Create Subscription')}
              </button>
            </div>
          </Form>
        </>
      )}
    </div>
  );
}
