// // src/components/adminpages/Subscriptions/Invoices.js
// import React from 'react';
// import CrudModule from '../../common/CRUD/CrudModule';
// import useCommonEnums from "../../common/CommonData";
// import AutoCompleteWidget from '../../common/Builder/AutoCompleteWidget';

// const SaaSInvoices = ({ isGlobalAccess, accountId }) => {
//   const enums = useCommonEnums();
//   const { InvoiceStatus = [] } = enums;
//   if (!InvoiceStatus.length) return <div>Loading form…</div>;

//   const today   = new Date().toISOString().split("T")[0];
//   const dueDate = new Date(Date.now() + 7*24*60*60*1000)
//                     .toISOString().split("T")[0];

//   const accountApi = '/accounts/';
  
//   const schema = {
//     title: 'Create SaaS Invoice',
//     type: 'object',
//     required: [
//       'account_id','invoice_title','amount','currency',
//       'status','invoice_date','due_date'
//     ],
//     properties: {
//       account_id:          { type:'string', title:'Account' },
//       invoice_title:       { type:'string', title:'Invoice Title' },
//       invoice_description: { type:'string', title:'Description' },
//       amount:              { type:'number', title:'Amount' },
//       currency:            {
//                               type:'string',
//                               title:'Currency',
//                               enum:['USD','INR','EUR'],
//                               default:'USD'
//                             },
//       status:              {
//                               type:'string',
//                               title:'Status',
//                               enum:InvoiceStatus
//                             },
//       invoice_date:        {
//                               type:'string',
//                               format:'date',
//                               title:'Invoice Date',
//                               default:today
//                             },
//       due_date:            {
//                               type:'string',
//                               format:'date',
//                               title:'Due Date',
//                               default:dueDate
//                             },
//       user_id:             {
//                               type:'string',
//                               title:'User ID',
//                               default: localStorage.getItem('user_id')||''
//                             }
//     }
//   };

//   const uiSchema = {
//     account_id: {
//       'ui:widget': AutoCompleteWidget,
//       'ui:options': {
//         api: accountApi,
//         listKey: 'accounts',       // matches your response
//         searchField: 'q',
//         labelField: 'account_name',
//         valueField: 'account_id',
//         debounce: 300,
//         placeholder: 'Search account…'
//       }
//     },
//     invoice_title:       { 'ui:placeholder': 'e.g. Q1 Subscription' },
//     invoice_description: { 'ui:widget':'textarea','ui:options':{rows:3} },
//     amount:              { 'ui:widget':'text','ui:placeholder':'Amount' },
//     currency:            { 'ui:widget':'select' },
//     status:              { 'ui:widget':'select' },
//     invoice_date:        { 'ui:widget':'date' },
//     due_date:            { 'ui:widget':'date' },
//     user_id:             { 'ui:widget':'hidden' }
//   };

//   const headerMapping = {
//     'Account':       { key: "account_id", clickable: true ,sortable: true},
//     'Title':         { key: "invoice_title", clickable: true ,sortable: true},
//     'Description':   { key: "invoice_description", clickable: false ,sortable: false},
//     'Amount':        { key: "amount", clickable: false ,sortable: false},
//     'Currency':      { key: "currency", clickable: false ,sortable: false},
//     'Status':        { key: "status", clickable: false ,sortable: true},
//     'Invoice Date':  { key: "invoice_date", clickable: false ,sortable: true},
//     'Due Date':      { key: "due_date", clickable: false ,sortable: true},
//     'Created On':    { key: "created_date", clickable: false ,sortable: false},
//   };

//   return (
//     <CrudModule
//       collectionName="saasinvoices"
//       title="SaaS Invoices"
//       description="Manage SaaS billing invoices"
//       headerMapping={headerMapping}
//       dataKey="items"
//       identifierKey="saasinvoices_id"
//       schema={schema}
//       uiSchema={uiSchema}
//       widgets={{ AutoCompleteWidget }}
//       fetchUrl="/crud/saasinvoices/"
//       searchFields={[
//         { name:'q', type:'text', placeholder:'Search invoices…' },
//         {
//           name:'status',
//           type:'dropdown',
//           placeholder:'Filter by status',
//           options:[
//             { label:'All', value:'' },
//             ...InvoiceStatus.map(s=>({label:s,value:s.toLowerCase()}))
//           ]
//         }
//       ]}
//       rowsPerPage={10}
//       actionFormats={[
//         { label:'view',  modal:true, url:'/saasinvoice/view/{saasinvoice_id}' },
//         { label:'edit',  modal:true, url:'/saasinvoice/update/{saasinvoice_id}' },
//         { label:'delete',modal:true, url:'/saasinvoice/delete/{saasinvoice_id}' }
//       ]}
//     />
//   );
// };

// export default SaaSInvoices;



import React from 'react';
import CrudModule from '../../common/CRUD/CrudModule';
import useCommonEnums from "../../common/CommonData";
import AutoCompleteWidget from '../../common/Builder/AutoCompleteWidget';

const SaaSInvoices = ({ isGlobalAccess, accountId }) => {
  // First, define all variables and states
  const { enums, loading, error } = useCommonEnums();
  const storedAccountId = localStorage.getItem('account_id') || '';
  const finalAccountId = accountId || storedAccountId;
  const userId = localStorage.getItem('user_id') || '';
  const today = new Date().toISOString().split("T")[0];
  const dueDate = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000).toISOString().split("T")[0];
  const accountApi = '/accounts/';

  // Default invoice status options
  const defaultInvoiceStatus = ['Draft', 'Sent', 'Paid', 'Overdue', 'Cancelled'];
  const InvoiceStatus = enums?.InvoiceStatus || enums?.invoice_status || defaultInvoiceStatus;

  // Define schema
  const schema = {
    title: 'Create SaaS Invoice',
    type: 'object',
    required: [
      'account_id', 'invoice_title', 'amount', 'currency',
      'status', 'invoice_date', 'due_date', 'invoice_type'
    ],
    properties: {
      account_id: { type: 'string', title: 'Account' },
      invoice_title: { type: 'string', title: 'Invoice Title' },
      invoice_description: { type: 'string', title: 'Description' },
      amount: { type: 'number', title: 'Amount' },
      currency: {
        type: 'string',
        title: 'Currency',
        enum: ['USD', 'INR', 'EUR'],
        default: 'USD'
      },
      status: {
        type: 'string',
        title: 'Status',
        enum: InvoiceStatus,
        default: InvoiceStatus[0] || 'Draft'
      },
      invoice_date: {
        type: 'string',
        format: 'date',
        title: 'Invoice Date',
        default: today
      },
      due_date: {
        type: 'string',
        format: 'date',
        title: 'Due Date',
        default: dueDate
      },
      user_id: {
        type: 'string',
        title: 'User ID',
        default: userId
      },
      invoice_type: {
        type: 'string',
        title: 'Invoice Type',
        enum: ['One-Time', 'Recurring', 'Custom'],
        default: 'One-Time'
      }
    }
  };

  // Define uiSchema
  const uiSchema = {
    account_id: {
      'ui:widget': AutoCompleteWidget,
      'ui:options': {
        api: accountApi,
        listKey: 'accounts',
        searchField: 'q',
        labelField: 'account_name',
        valueField: 'account_id',
        debounce: 300,
        placeholder: 'Search account…'
      }
    },
    invoice_title: { 'ui:placeholder': 'e.g. Q1 Subscription' },
    invoice_description: { 'ui:widget': 'textarea', 'ui:options': { rows: 3 } },
    amount: { 'ui:widget': 'text', 'ui:placeholder': 'Amount' },
    currency: { 'ui:widget': 'select' },
    status: { 'ui:widget': 'select' },
    invoice_date: { 'ui:widget': 'date' },
    due_date: { 'ui:widget': 'date' },
    user_id: { 'ui:widget': 'hidden' },
    invoice_type: { 'ui:widget': 'select', 'ui:placeholder': 'Select invoice type' }
  };

  // Define headerMapping
  const headerMapping = {
    'Account': { key: "account_id", clickable: true, sortable: true },
    'Title': { key: "invoice_title", clickable: true, sortable: true },
    'Description': { key: "invoice_description", clickable: false, sortable: false },
    'Amount': { key: "amount", clickable: false, sortable: false },
    'Currency': { key: "currency", clickable: false, sortable: false },
    'Status': { key: "status", clickable: false, sortable: true },
    'Invoice Type': { key: "invoice_type", clickable: false, sortable: true },
    'Invoice Date': { key: "invoice_date", clickable: false, sortable: true },
    'Due Date': { key: "due_date", clickable: false, sortable: true },
    'Created On': { key: "created_date", clickable: false, sortable: false },
  };

  // Loading and error states
  if (loading) return <div>Loading form data...</div>;
  if (error && InvoiceStatus.length === 0) {
    console.error('Error loading enums:', error);
    return <div className="text-red-500">Error loading invoice status data. Using default options.</div>;
  }

  return (
    <CrudModule
      collectionName="saasinvoices"
      title="SaaS Invoices"
      description="Manage SaaS billing invoices"
      headerMapping={headerMapping}
      dataKey="items"
      identifierKey="saasinvoices_id"
      schema={schema}
      uiSchema={uiSchema}
      widgets={{ AutoCompleteWidget }}
      fetchUrl={`/invoices/?account_id=${finalAccountId}`}
      searchFields={[
        { name: 'q', type: 'text', placeholder: 'Search invoices…' },
        // {
        //   name: 'status',
        //   type: 'dropdown',
        //   placeholder: 'Filter by status',
        //   options: [
        //     { label: 'All', value: '' },
        //     ...InvoiceStatus.map(s => ({ label: s, value: s.toLowerCase() }))
        //   ]
        // }
      ]}
      rowsPerPage={10}
      actionFormats={[
        { label: 'view', modal: true, url: '/saasinvoice/view/{saasinvoice_id}' },
        { label: 'edit', modal: true, url: '/saasinvoice/update/{saasinvoice_id}' },
        { label: 'delete', modal: true, url: '/saasinvoice/delete/{saasinvoice_id}' }
      ]}
    />
  );
};

export default SaaSInvoices;