import React, { useState } from 'react';
import CrudModule from '../common/CRUD/CrudModule';

const skillSchema = {
  title: "Skill",
  type: "object",
  required: ["name"],
  properties: {
    name: { 
      type: "string", 
      title: "Skill Name",
      style: { width: '250px' } // Added width
    },
  }
};

const skillUiSchema = {
  name: { 
    "ui:placeholder": "Enter skill name",
    "ui:options": {
      style: { width: '650px' } // Added width
    }
  },
  status: {
    "ui:widget": "select",
    "ui:options": {
      style: { width: '250px' }, // Added width
      enumOptions: [
        { value: "active", label: "Active" },
        { value: "inactive", label: "Inactive" },
        { value: "on_leave", label: "On Leave" }
      ]
    }
  }
};

const Skills = () => {
  const collectionName = "skills";
  const rowsPerPage = 10;

  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: '',
    status: ''
  });

  const formatDate = (dateString) => {
    if (!dateString) return '—';
    try {
      return new Date(dateString).toLocaleString();
    } catch (e) {
      return dateString;
    }
  };

  const buildFetchUrl = () => {
    const query = new URLSearchParams();
    query.append('skip', searchParams.skip);
    query.append('limit', searchParams.limit);
    if (searchParams.q) query.append('q', searchParams.q);
    if (searchParams.status) query.append('status', searchParams.status);
    return `/crud/${collectionName}/?${query.toString()}`;
  };

  const handleSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      status: params.status || '',
      skip: 0
    }));
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  // Styles object
  const styles = {
    searchInput: {
      width: '250px', // Consistent width
      padding: '8px 12px',
      borderRadius: '4px',
      border: '1px solid #ccc'
    }
  };

  return (
    <CrudModule
      collectionName={collectionName}
      title="Skills"
      description="Manage skills in the system."
      headerMapping={{
        "Skill Name": {
          key: "name",
          default: "—"
        },
        "Created Date": {
          key: "created_date",
          render: (value) => formatDate(value)
        }
      }}
      dataKey="items"
      identifierKey="skills_id"
      schema={skillSchema}
      uiSchema={skillUiSchema}
      fetchUrl={buildFetchUrl()}
      searchFields={[
        {
          name: 'q',
          type: 'text',
          placeholder: 'Search skills...',
          defaultValue: searchParams.q,
          style: styles.searchInput
        },
        {
          name: 'status',
          type: 'dropdown',
          placeholder: 'Status',
          options: [
            { label: 'All', value: '' },
            { label: 'Active', value: 'active' },
            { label: 'Inactive', value: 'inactive' },
            { label: 'On Leave', value: 'on_leave' }
          ],
          defaultValue: searchParams.status,
          style: { width: '250px' } // Consistent width
        }
      ]}
      rowsPerPage={rowsPerPage}
      actionFormats={[
        { label: "view", modal: true, url: "/skills/view/{skills_id}" },
        { label: "edit", modal: true, url: "/skills/update/{skills_id}" },
        { label: "delete", modal: true, url: "/skills/delete/{skills_id}" },
      ]}
      onSearch={handleSearch}
      onPageChange={handlePageChange}
      emptyMessage="No skills found."
      customRowRenderer={(row) => ({
        ...row,
        created_date: formatDate(row.created_date)
      })}
    />
  );
};

export default Skills;