// src/components/adminpages/Settings/SettingsPage.js
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Tabs from '@mui/material/Tabs';
import Tab from '@mui/material/Tab';
import Typography from '@mui/material/Typography';
import Divider from '@mui/material/Divider';
import Paper from '@mui/material/Paper';

// ✅ use the actual files you have
import ApplicationSettings from './ApplicationSettingsPage';
import PaymentSettings from './PaymentSettingsPage';
import CommunicationSettings from './CommunicationSettingsPage';
import ReferralSettings from './ReferralSettingsPage';

const SettingsPage = () => {
  const [activeTab, setActiveTab] = useState(0);
  const [loading, setLoading] = useState({});
  const [settingsData, setSettingsData] = useState({});

  // map each tab index to its API endpoint
  // (match what components send in onSave: 'referral' | 'application' | 'communication' | 'payment')
  const tabEndpoints = {
    0: 'referral',
    1: 'application',
    2: 'communication',
    3: 'payment',
  };

  const fetchTabData = async (tabIndex) => {
    const endpoint = tabEndpoints[tabIndex];
    if (!endpoint) return;

    setLoading(prev => ({ ...prev, [endpoint]: true }));
    try {
      const accountId = localStorage.getItem('account_id');
      const { data } = await axios.get(`${endpoint}/${accountId}`);
      setSettingsData(prev => ({ ...prev, [endpoint]: data }));
    } catch (err) {
      console.error(`Error fetching ${endpoint} settings:`, err);
    } finally {
      setLoading(prev => ({ ...prev, [endpoint]: false }));
    }
  };

  useEffect(() => {
    fetchTabData(activeTab);
  }, []); // initial load for tab 0

  const handleTabChange = (_e, newVal) => {
    setActiveTab(newVal);
    fetchTabData(newVal);
  };

  const handleSaveSettings = async (section, data) => {
    try {
      const accountId = localStorage.getItem('account_id');
      await axios.post(`${section}/${accountId}`, data);
      setSettingsData(prev => ({ ...prev, [section]: data }));
      alert(`${section.replace('_', ' ')} settings updated successfully!`);
    } catch (err) {
      console.error(`Error updating ${section} settings:`, err);
      alert(`Failed to update ${section} settings`);
    }
  };

  const tabs = [
    { label: 'Referral',       icon: 'diversity_3',    component: ReferralSettings },
    { label: 'Application',    icon: 'map',            component: ApplicationSettings },
    { label: 'Communication',  icon: 'mail',           component: CommunicationSettings },
    { label: 'Payment',        icon: 'payments',       component: PaymentSettings },
  ];

  const currentEndpoint = tabEndpoints[activeTab];
  const isTabLoading = loading[currentEndpoint];

  return (
    <div style={{ display: 'flex', minHeight: '100vh', overflowY: 'auto', marginTop: 16 }}>
      <Paper elevation={2} style={{ width: 250, minWidth: 250, maxWidth: 250, marginRight: 16, background: '#fff', flexShrink: 0 }}>
        <Tabs
          orientation="vertical"
          variant="scrollable"
          value={activeTab}
          onChange={handleTabChange}
          aria-label="Settings vertical tabs"
          TabIndicatorProps={{ style: { backgroundColor: '#34c759', width: 4, left: 0 } }}
          style={{ height: '100%' }}
        >
          {tabs.map((tab, i) => {
            const isActive = activeTab === i;
            const tabIsLoading = loading[tabEndpoints[i]];
            return (
              <Tab
                key={i}
                label={
                  <div style={{ display: 'flex', alignItems: 'center', gap: 8 }}>
                    <span className="material-symbols-outlined" style={{ color: isActive ? 'green' : 'black' }}>{tab.icon}</span>
                    <Typography variant="body2" style={{ color: isActive ? 'green' : 'black' }}>
                      {tab.label}{tabIsLoading && ' (Loading...)'}
                    </Typography>
                  </div>
                }
                style={{
                  alignItems: 'flex-start',
                  justifyContent: 'flex-start',
                  textTransform: 'none',
                  backgroundColor: isActive ? '#ffffff' : 'transparent',
                  padding: '8px 16px',
                  width: '100%',
                  maxWidth: '100%',
                  minHeight: 48,
                }}
              />
            );
          })}
        </Tabs>
      </Paper>

      <div style={{ flexGrow: 1, padding: 16 }}>
        <Typography variant="h6" gutterBottom>
          {tabs[activeTab].label}{isTabLoading && ' (Loading...)'}
        </Typography>
        <Divider style={{ marginBottom: 16 }} />
        {isTabLoading
          ? <div>Loading {tabs[activeTab].label.toLowerCase()} settings...</div>
          : React.createElement(tabs[activeTab].component, { data: settingsData, onSave: handleSaveSettings })}
      </div>
    </div>
  );
};

export default SettingsPage;
