import React, { useState, useEffect } from 'react';
import axios from 'axios';
import ServicesForm from './ServicesForm';
import Modal from '../../Layouts/Modal';

const ServicesUpdate = ({ userId, onClose, onUpdateSuccess }) => {  
  const [serviceData, setServiceData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    axios.get(`services/${userId}`)
      .then(res => {
        // Transform the API data to match form structure
        const transformedData = {
          service_title: res.data.service_title,
          service_description: res.data.service_description,
          skills: res.data.skills,
          addons: res.data.addons,
          checklist: res.data.checklist,
          capacity_jobs: res.data.capacity_jobs,
          // Convert boolean values to 'yes'/'no' strings for radio buttons
          has_cost: res.data.has_cost ? 'yes' : 'no',
          is_vehicle_focused: res.data.is_vehicle_focused ? 'yes' : 'no',
          image: res.data.image,
          map_icon: res.data.map_icon,
          category_inputs: res.data.category_inputs,
          dynamicFields: res.data.dynamicFields
        };
        setServiceData(transformedData);
        setLoading(false);
      })
      .catch(err => {
        setError("Error fetching service data.");
        setLoading(false);
        console.error("API Error:", err);
      });
  }, [userId]);

  const handleSubmit = async (formData) => {
    try {
      const formDataToSend = new FormData();
      
      // Append all non-file fields, converting yes/no back to booleans
      Object.keys(formData).forEach(key => {
        if (key !== 'image' && key !== 'map_icon') {
          // Convert 'yes'/'no' back to boolean for API
          let value = formData[key];
          if (key === 'has_cost' || key === 'is_vehicle_focused') {
            value = value === 'yes';
          }
          formDataToSend.append(key, value);
        }
      });

      // Handle image update if it's a new file (data URL)
      if (formData.image && formData.image.startsWith('data:')) {
        const imageFile = dataURLtoFile(formData.image, 'service-image');
        formDataToSend.append('image', imageFile);
      }

      // Handle map icon update if it's a new file (data URL)
      if (formData.map_icon && formData.map_icon.startsWith('data:')) {
        const iconFile = dataURLtoFile(formData.map_icon, 'map-icon');
        formDataToSend.append('map_icon', iconFile);
      }

      const response = await axios.put(
        `services/${userId}`,
        formDataToSend,
        {
          headers: {
            'Content-Type': 'multipart/form-data'
          }
        }
      );
      
      alert("Service updated successfully!");
      if (onUpdateSuccess) onUpdateSuccess(response.data);
      onClose();
    } catch (err) {
      console.error("Error updating service:", err);
      alert("Error updating service.");
    }
  };

  // Helper function to convert data URL to File object
  const dataURLtoFile = (dataurl, filename) => {
    const arr = dataurl.split(',');
    const mime = arr[0].match(/:(.*?);/)[1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8arr = new Uint8Array(n);
    while (n--) {
      u8arr[n] = bstr.charCodeAt(n);
    }
    return new File([u8arr], filename, { type: mime });
  };

  return (
    <Modal title="Edit Service" onClose={onClose}>
      {loading ? (
        <div>Loading service data...</div>
      ) : error ? (
        <div style={{ color: 'red' }}>{error}</div>
      ) : serviceData ? (
        <ServicesForm 
          initialData={serviceData}
          onSubmit={handleSubmit}
          isEditMode={true}
        />
      ) : (
        <div>No service data found</div>
      )}
    </Modal>
  );
};

export default ServicesUpdate;