import React, { useState, useEffect } from 'react';
import axios from 'axios';
import RoleForm from './RoleForm';
import Modal from '../../Layouts/Modal';

const RolesUpdate = ({ roleId, onClose, onUpdateSuccess }) => {
  const [roleData, setRoleData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState(false);
  const [message, setMessage] = useState('');

  // Helper to convert backend permissions (array of actions) into the boolean object expected by RoleForm.
  const transformPermissionsForForm = (permissions) => {
    const defaultActions = { create: false, read: false, update: false, delete: false };
    const newPermissions = {};
    if (!permissions) return newPermissions;
    Object.keys(permissions).forEach(featureKey => {
      newPermissions[featureKey] = { ...defaultActions };
      const allowedActions = permissions[featureKey]; // e.g., ["create", "read"]
      allowedActions.forEach(action => {
        newPermissions[featureKey][action] = true;
      });
    });
    return newPermissions;
  };

  useEffect(() => {
    axios.get(`/roles/${roleId}`)
      .then(res => {
        let data = res.data;
        console.log("Fetched role data:", data);
        // Convert potential string booleans to actual booleans.
        data.is_global_access = data.is_global_access === true || data.is_global_access === "true";
        data.is_saas_only = data.is_saas_only === true || data.is_saas_only === "true";
        if (data.is_system_default !== undefined) {
          data.is_system_default = data.is_system_default === true || data.is_system_default === "true";
        }
        if (data.permissions) {
          data.permissions = transformPermissionsForForm(data.permissions);
        }
        setRoleData(data);
        setLoading(false);
      })
      .catch(err => {
        setError("Error fetching role data.");
        setLoading(false);
      });
  }, [roleId]);

  // Note: Since RoleForm already transforms permissions on submit,
  // we simply pass the received data to the API without re-transforming.
  const handleSubmit = (data) => {
    axios.put(`/roles/${roleId}`, data)
      .then(res => {
        setSuccess(true);
        setMessage("Role updated successfully!");
        if (onUpdateSuccess) onUpdateSuccess(res.data);
        // Automatically close the modal after 2 seconds.
        setTimeout(() => {
          onClose();
        }, 2000);
      })
      .catch(err => {
        setError("Error updating role.");
      });
  };

  return (
    <Modal title="Edit Role" onClose={onClose}>
      {loading ? (
        <div>Loading role data...</div>
      ) : error ? (
        <div style={{ color: 'red' }}>{error}</div>
      ) : success ? (
        <div className="success">{message}</div>
      ) : (
        <RoleForm initialData={roleData} onSubmit={handleSubmit} />
      )}
    </Modal>
  );
};

export default RolesUpdate;
