import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import RoleView from './RoleView';
import RolesUpdate from './RolesUpdate';
import RoleDelete from './RoleDB';

const ROLES_HEADER_TO_KEY_MAP = {
  "Role Name": { 
    key: "name",
    label: "Role Name",
    clickable: true,
    sortable: true,
    formatter: (value) => value || '-'
  },
  "Description": { 
    key: "description",
    label: "Description",
    sortable: true,
    formatter: (value) => value || "No description" 
  },
  "Created Date": { 
    key: "created_date",
    label: "Created Date",
    sortable: true,
    formatter: (value) => value ? new Date(value).toLocaleDateString() : "N/A" 
  },
  "Updated Date": { 
    key: "updated_date",
    label: "Updated Date",
    sortable: true,
    formatter: (value) => value ? new Date(value).toLocaleDateString() : "N/A" 
  },
};

const RolesList = ({ initialSearchQuery = '', rowsPerPage = 10, isDefaultRoles = false }) => {
  const fetchDataUrl = isDefaultRoles ? `/roles/d/` : `/roles/`;
  const [selectedRole, setSelectedRole] = useState(null);

  const defaultActionFormats = [
    { label: "view", modal: true, url: '/' },
  ];

  const customActionFormats = [
    { label: "view", modal: true, url: "/roles/update/{id}" },
    { label: "edit", modal: true, url: "/roles/update/{id}" },
    { label: "update", modal: true, url: "/roles/update/{id}" },
    { label: "delete", modal: true, url: "/roles/delete/{id}" },
  ];

  // Define search fields for filtering
  const searchFields = [
    { 
      name: 'q', 
      type: 'text', 
      placeholder: 'Search...', 
      defaultValue: initialSearchQuery 
    },
    { 
      name: 'status', 
      type: 'dropdown', 
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' },
        { label: 'Pending', value: 'pending' },
      ]
    },
    { 
      name: 'created_date_from', 
      type: 'date', 
      placeholder: 'From Date' 
    },
    { 
      name: 'created_date_to', 
      type: 'date', 
      placeholder: 'To Date' 
    },
   
  ];

  const handleActionClick = (roleId, action) => {
    if (action.label === "view") {
      setSelectedRole({ roleId, modalType: "view" });
    } else if (action.label === "edit" || action.label === "update") {
      setSelectedRole({ roleId, modalType: "update" });
    } else if (action.label === "delete") {
      setSelectedRole({ roleId, modalType: "delete" });
    }
  };

  const handleCellClick = (header, item) => {
    if (header === "Role Name") {
      setSelectedRole({
        roleId: item.role_id,
        modalType: "view",
        data: item
      });
    }
  };

  const closeModal = () => setSelectedRole(null);

  return (
    <div>
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={ROLES_HEADER_TO_KEY_MAP}
        dataKey="roles"
        identifierKey="role_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={isDefaultRoles ? defaultActionFormats : customActionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        searchFields={searchFields}
      />

      {selectedRole?.modalType === "view" && (
        <RoleView
          roleId={selectedRole.roleId}
          roleData={selectedRole.data}
          onClose={closeModal}
          onUpdateSuccess={() => console.log("Role viewed")}
        />
      )}

      {!isDefaultRoles && selectedRole?.modalType === "update" && (
        <RolesUpdate
          roleId={selectedRole.roleId}
          onClose={closeModal}
          onUpdateSuccess={() => console.log("Role updated")}
        />
      )}

      {!isDefaultRoles && selectedRole?.modalType === "delete" && (
        <RoleDelete
          roleId={selectedRole.roleId}
          onClose={closeModal}
          onDeleteSuccess={() => console.log("Role deleted")}
        />
      )}
    </div>
  );
};

export default RolesList;