
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';

const TAB_CONFIG = [
  { id: 'dashboard', label: 'View Role', icon: 'grid_view' },
];

const RoleView = ({ roleId, onClose }) => {
  const [roleData, setRoleData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);

  useEffect(() => {
    axios.get(`/roles/${roleId}`)
      .then(res => {
        setRoleData(res.data);
        setLoading(false);
      })
      .catch(err => {
        setError('Error fetching role details.');
        setLoading(false);
      });
  }, [roleId]);

  if (loading) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', textAlign: 'center' }}>Loading role data...</div>
      </Modal>
    );
  }

  if (error) {
    return (
      <Modal onClose={onClose}>
        <div style={{ padding: '20px', color: '#f44336' }}>{error}</div>
      </Modal>
    );
  }

  const renderPermissions = (permissions) => {
    if (!permissions || typeof permissions !== 'object') return null;
    return Object.entries(permissions).map(([moduleId, actions]) => (
      <div key={moduleId}>
        <p style={{ margin: 0 }}>
          <strong>{moduleId}:</strong> {actions.join(', ')}
        </p>
      </div>
    ));
  };

  return (
    <Modal onClose={onClose}>
      <div style={{ height: '100%', overflow: 'hidden' }}>
        {/* Header */}
        <div style={{
          display: 'flex',
          alignItems: 'center',
          gap: '20px',
          padding: '0px 10px 10px',
          borderBottom: '1px solid #e0e0e0'
        }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold'
          }}>
            {roleData.name?.charAt(0)?.toUpperCase() || 'R'}
          </div>
          <div>
            <h3 style={{ margin: '0px', fontWeight: 'bold', fontSize: '20px', color: 'black' }}>
              {roleData.name}
            </h3>
            <p style={{ marginTop: '5px', fontSize: '16px', color: 'black', fontWeight: 'bold' }}>
              ID : # {roleData.role_id}

            </p>
          </div>
        </div>

        {/* Tabs */}
        <div style={{
          display: 'flex',
          borderBottom: '1px solid #e0e0e0',
          marginBottom: '20px',
          padding: '20px 6px'
        }}>
          {TAB_CONFIG.map(tab => (
            <button
              key={tab.id}
              style={{
                padding: '2px 16px',
                border: 'none',
                backgroundColor: 'transparent',
                cursor: 'pointer',
                fontSize: '14px',
                fontWeight: activeTab === tab.id ? '400' : '400',
                color: activeTab === tab.id ? '#34c759' : '#333',
                position: 'relative',
                whiteSpace: 'nowrap',
                minWidth: '100px',
                textAlign: 'center',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                gap: '8px'
              }}
              onClick={() => setActiveTab(tab.id)}
              onMouseEnter={() => setHoveredTab(tab.id)}
              onMouseLeave={() => setHoveredTab(null)}
            >
              <span className="material-symbols-outlined" style={{
                color: activeTab === tab.id ? '#34c759' : '#333',
                fontSize: '20px'
              }}>
                {tab.icon}
              </span>
              {(activeTab === tab.id || hoveredTab === tab.id) && <span>{tab.label}</span>}
              {activeTab === tab.id && (
                <div style={{
                  position: 'absolute',
                  bottom: '-20px',
                  left: '0',
                  right: '0',
                  height: '2px',
                  backgroundColor: '#34c759'
                }} />
              )}
            </button>
          ))}
        </div>

        {/* Content */}
        <div style={{ padding: '0 20px' }}>
          <p><strong>Description:</strong> {roleData.description}</p>
          <p><strong>Created Date:</strong> {new Date(roleData.created_date).toLocaleString()}</p>
          <p><strong>Updated Date:</strong> {new Date(roleData.updated_date).toLocaleString()}</p>
          <p><strong>Account Specific:</strong> {roleData.account_id ? roleData.account_id : 'Global'}</p>
          <p><strong>System Default:</strong> {roleData.is_system_default ? 'Yes' : 'No'}</p>
          <p><strong>Global Access:</strong> {roleData.is_global_access ? 'Yes' : 'No'}</p>
          <p><strong>SaaS Only:</strong> {roleData.is_saas_only ? 'Yes' : 'No'}</p>
          <h4 style={{ marginTop: "1rem" }}>Permissions:</h4>
          {renderPermissions(roleData.permissions)}
        </div>
      </div>
    </Modal>
  );
};

export default RoleView;
