import React, { useState, useEffect } from 'react';
import { withTheme } from '@rjsf/core';
import MaterialUITheme from '@rjsf/material-ui';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';
import { Grid } from '@material-ui/core';

const Form = withTheme(MaterialUITheme);

// Custom field for rendering the "permissions" property
const PermissionsField = (props) => {
  const { formData = {}, onChange, schema } = props;
  const modules = schema.properties || {};

  const handleMasterChange = (e) => {
    const checked = e.target.checked;
    const newData = {};
    Object.keys(modules).forEach(moduleKey => {
      newData[moduleKey] = {};
      Object.keys(modules[moduleKey].properties).forEach(action => {
        newData[moduleKey][action] = checked;
      });
    });
    onChange(newData);
  };

  let allChecked = true;
  Object.keys(modules).forEach(moduleKey => {
    const moduleData = formData[moduleKey] || {};
    Object.keys(modules[moduleKey].properties).forEach(action => {
      if (!moduleData[action]) {
        allChecked = false;
      }
    });
  });

  const handleCheckboxChange = (moduleKey, action) => (e) => {
    const checked = e.target.checked;
    const newData = { ...formData };
    newData[moduleKey] = { ...(newData[moduleKey] || {}) };
    newData[moduleKey][action] = checked;
    onChange(newData);
  };

  return (
    <div>
      {/* Table Header */}
      <Grid container spacing={2} alignItems="center">
        <Grid item xs={3}>
          <strong>Feature</strong>
        </Grid>
        <Grid item xs={9}>
          <Grid container spacing={2} alignItems="center">
            <Grid item xs={3}><strong>Create</strong></Grid>
            <Grid item xs={3}><strong>Read</strong></Grid>
            <Grid item xs={3}><strong>Update</strong></Grid>
            <Grid item xs={3}><strong>Delete</strong></Grid>
          </Grid>
        </Grid>
      </Grid>
      {/* Master Toggle Row */}
      <Grid container spacing={2} alignItems="center" style={{ marginBottom: 16 }}>
        <Grid item xs={3}>
          <strong>All</strong>
        </Grid>
        <Grid item xs={9}>
          <input type="checkbox" checked={allChecked} onChange={handleMasterChange} />
        </Grid>
      </Grid>
      {/* Render each feature row */}
      {Object.keys(modules).map(moduleKey => {
        const moduleSchema = modules[moduleKey];
        const moduleData = formData[moduleKey] || {};
        return (
          <Grid container spacing={2} alignItems="center" key={moduleKey} style={{ marginBottom: 5, borderBottom: '1px solid #eee' }}>
            <Grid item xs={3}>
              {moduleSchema.title || moduleKey}
            </Grid>
            <Grid item xs={9}>
              <Grid container spacing={2} alignItems="center">
                {['create', 'read', 'update', 'delete'].map(action => (
                  <Grid item xs={3} key={action}>
                    <input 
                      type="checkbox" 
                      checked={moduleData[action] || false} 
                      onChange={handleCheckboxChange(moduleKey, action)}
                    />
                    <span style={{ marginLeft: 8 }}>
                      {moduleSchema.properties[action].title}
                    </span>
                  </Grid>
                ))}
              </Grid>
            </Grid>
          </Grid>
        );
      })}
      <p></p>
    </div>
  );
};

const RoleForm = ({ initialData, onSubmit }) => {
  const [formData, setFormData] = useState(initialData || {});
  const [schema, setSchema] = useState({});
  const [uiSchema, setUiSchema] = useState({});

  // Sync the form data whenever initialData changes.
  useEffect(() => {
    setFormData(initialData || {});
  }, [initialData]);

  useEffect(() => {
    const userRole = parseInt(localStorage.getItem('roles'), 10);
    const accountId = localStorage.getItem('account_id') || "";

    axios.get('/crud/features?limit=100')
      .then(res => {
        const items = res.data.items || [];
        // Use is_saas_only from formData if available; otherwise, default based on userRole.
        const isSaasOnly = formData.is_saas_only !== undefined ? formData.is_saas_only : (userRole === 1);
        const features = isSaasOnly 
          ? items.filter(f => f.category === 'saas')
          : items.filter(f => f.category === 'platform');

        // Build permission properties using feature.features_id as the key.
        const permissionProperties = features.reduce((acc, feature) => {
          acc[feature.features_id] = {
            type: "object",
            title: feature.name,
            properties: {
              create: { type: "boolean", title: "Create", default: false },
              read: { type: "boolean", title: "Read", default: false },
              update: { type: "boolean", title: "Update", default: false },
              delete: { type: "boolean", title: "Delete", default: false }
            }
          };
          return acc;
        }, {});

        // Build baseProperties with desired fields.
        const baseProperties = {
          name: { type: "string", title: "Role Name" },
          description: { type: "string", title: "Description" },
          account_id: { type: "string", title: "Account ID", default: userRole === 1 ? "" : accountId },
          is_saas_only: { type: "boolean", title: "SaaS Only", default: userRole === 1 ? true : false },
          ...(userRole === 1 && { is_system_default: { type: "boolean", title: "System Default", default: false } }),
          is_global_access: { type: "boolean", title: "Is Global Access?", default: false },
          permissions: {
            type: "object",
            title: "Permissions",
            properties: permissionProperties
          }
        };

        setSchema({
          title: "",
          type: "object",
          required: ["name"],
          properties: baseProperties
        });

        // Build uiSchema based on user role.
        const ui = {
          permissions: { "ui:field": "permissionsField" },
          account_id: { "ui:widget": "hidden" }
        };
        if (userRole !== 1) {
          ui.is_saas_only = { "ui:widget": "hidden" };
          ui.is_system_default = { "ui:widget": "hidden" };
        }
        setUiSchema(ui);
      })
      .catch(err => console.error("Error fetching features", err));
  }, [formData.is_saas_only]);

  // Transform permission booleans into an array of allowed actions.
  const transformPermissions = (permissions) => {
    const newPermissions = {};
    if (!permissions) return newPermissions;
    Object.keys(permissions).forEach(featureKey => {
      const actions = permissions[featureKey];
      if (actions && typeof actions === 'object') {
        newPermissions[featureKey] = Object.keys(actions).filter(action => actions[action]);
      }
    });
    return newPermissions;
  };

  const handleSubmit = ({ formData }) => {
    const transformedData = {
      ...formData,
      permissions: transformPermissions(formData.permissions)
    };
    onSubmit(transformedData);
  };

  if (!schema.properties) {
    return <div>Loading...</div>;
  }

  return (
    <Form
      schema={schema}
      uiSchema={uiSchema}
      formData={formData}
      onChange={({ formData }) => setFormData(formData)}
      onSubmit={handleSubmit}
      validator={validator}
      showErrorList={false}
      fields={{ permissionsField: PermissionsField }}
    />
  );
};

export default RoleForm;
